/* $Header: /home/cvs/bp/oofem/oofemlib/src/domain.C,v 1.31.4.1 2004/04/05 15:19:43 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/

//   file DOMAIN.CC

#include "domain.h"
#include "element.h"
#include "timestep.h"
#include "node.h"
#include "elementside.h"
#include "material.h"
#include "crosssection.h"
//#include "yieldcriteria.h"
#include "load.h"
#include "loadtime.h"
#include "engngm.h"
#ifndef __MAKEDEPEND
//#include <clock.h>
#endif
#include "clock.h"
#include "debug.h"
#include "verbose.h"
#include "strreader.h"
#include "cltypes.h"
#include "conTable.h"
#include "outputmanager.h"
#include "dummylocalizer.h"
#include "octreelocalizer.h"
#include "datareader.h"
#include "util.h"
#include "nodalrecoverymodel.h"
#include "nonlocalbarrier.h"
#include "usrdefsub.h"
#include "logger.h"

#ifdef __PARALLEL_MODE
#include "parallel.h"
#endif

#ifndef __MAKEDEPEND
#include <string.h>
#include <stdarg.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <ctype.h>
#endif

Domain :: Domain (int n, int serNum, EngngModel* pm) : defaultNodeDofIDArry(), defaultSideDofIDArry()
   // Constructor. Creates a new domain.
{
 this->engineeringModel = pm;
 this->number   = n;
 this->serialNumber = serNum;

 elementList           = new AList<Element>(0) ;
 dofManagerList        = new AList<DofManager>(0) ;
 materialList          = new AList<Material>(0) ;
 loadList              = new AList<Load>(0) ;
 loadTimeFunctionList  = new AList<LoadTimeFunction>(0) ;
 crossSectionList      = new AList<CrossSection>(0) ;
 nonlocalBarierList    = new AList<NonlocalBarrier>(0) ;
 // yieldCriteriaList     = new AList(0) ;

 numberOfElements      = 0 ;
 numberOfDefaultDofsPerNode = -1 ;
 numberOfDefaultDofsPerSide = -1 ;
 dType                 = _unknownMode ;
 
 connectivityTable     = NULL;
 spatialLocalizer      = NULL;
 outputManager         = new OutputManager (this);
 smoother = NULL;

 nonlocalUpdateStateCounter = 0;
}

Domain :: ~Domain ()
   // Destructor.
{
   delete elementList ;
   delete dofManagerList ;
   delete materialList ;
   delete loadList ;
   delete loadTimeFunctionList ;
   delete crossSectionList ;
  delete nonlocalBarierList ;

   delete connectivityTable;
  delete spatialLocalizer;
  delete outputManager;
  if (smoother) delete smoother;

}


Element*  Domain :: giveElement (int n)
   // Returns the n-th element. Generates error if it is not defined yet.
{

   if (elementList -> includes(n))
      return elementList->at(n) ;
   else {
     _error2 ("giveElement: undefined element (%d)",n);
     // elem = (Element*) Element(n,this).typed() ;
     // elementList -> put(n,elem) ;}
   }
   return NULL ;
}

/*
FILE*  Domain :: giveInputStream ()
   // Returns an input stream on the data file of the receiver.
{
   if (inputStream)
      return inputStream ;
   else {
   fprintf (stderr,"\nDomain->giveInputStream: Internal error\a\n") ;
   exit (1);}
  
  return inputStream ;
}

FILE*  Domain :: giveOutputStream ()
   // Returns an output stream on the data file of the receiver.
{
   if (! outputStream) {
   fprintf (stderr,"\nDomain->giveOutputStream: Internal error\a\n") ;
   exit (1);
  }
   return outputStream ;
}
*/
Load*  Domain :: giveLoad (int n)
   // Returns the n-th load. Generates the error if not defined.
{

   if (loadList -> includes(n))
      return loadList->at(n) ;
   else {
     _error2 ("giveLoad: undefined load (%d)",n);
//      load = (Load*) Load(n,this).typed() ;
//      loadList -> put(n,load) ;}
   }
   return NULL ;
}


LoadTimeFunction*  Domain :: giveLoadTimeFunction (int n)
   // Returns the n-th load-time function. Creates this fuction if it does
   // not exist yet.
{
   if (loadTimeFunctionList -> includes(n))
      return loadTimeFunctionList->at(n) ;
   else {
     _error2 ("giveLoadTimeFunction: undefined load-time function (%d)",n);
     //      ltf = (LoadTimeFunction*) LoadTimeFunction(n,this).typed() ;
     //      loadTimeFunctionList -> put(n,ltf) ;}
   }
   return NULL ;
}


Material*  Domain :: giveMaterial (int n)
   // Returns the n-th material. Creates this material if it does not exist
   // yet.
{

   if (materialList -> includes(n))
      return materialList -> at(n) ;
   else {
     _error2 ("giveMaterial: undefined material (%d)",n);
     //      mat = new Material(n,this) ;
     //      materialList  -> put(n,mat) ;}
   }
   return NULL ;
}


Node*  Domain :: giveNode (int n)
   // Returns the n-th node. Creates this node if it does not exist yet.
{
   DofManager *node=NULL ;

   if (dofManagerList -> includes(n)) {
      node = dofManagerList -> at(n) ;
   if ((node->giveClassID() != NodeClass) && (node->giveClassID() != RigidArmNodeClass) && (node->giveClassID() != HangingNodeClass)) {
    _error2 ("giveNode: incompatible type of dofManager %d, can not convert",n);
   }
  } else {
   _error2 ("giveNode: undefined dofManager (%d)",n);
   //      node = new Node(n,this) ;
   //      nodeList  -> put(n,node) ;}
  }
   return (Node*) node ;
 }

ElementSide*  Domain :: giveSide (int n)
   // Returns the n-th element side.
{
   DofManager *side = NULL ;

   if (dofManagerList -> includes(n)) {
      side = dofManagerList -> at(n) ;
   if (side->giveClassID() != ElementSideClass) {
    _error2 ("giveSide: incompatible type of dofManager %d, can not convert",n);
   }
  } else {
   _error2 ("giveSide: undefined dofManager (%d)",n);
  }
   return (ElementSide*) side ;
 }

DofManager*  Domain :: giveDofManager (int n)
   // Returns the n-th node. Creates this node if it does not exist yet.
{
   if (dofManagerList -> includes(n)) {
   return dofManagerList -> at(n) ;
  } else {
   _error2 ("giveDofManager: undefined dofManager (%d)",n);
   //      node = new Node(n,this) ;
   //      nodeList  -> put(n,node) ;}
  }
   return NULL ;
 }



CrossSection*  Domain :: giveCrossSection (int n)
   // Returns the n-th cross section.
   // yet.
{

   if (crossSectionList -> includes(n))
      return crossSectionList -> at(n) ;
   else {
     _error2 ("giveCrossSection: undefined cross section (%d)",n);
   }
   return NULL;
}


NonlocalBarrier*  Domain :: giveNonlocalBarrier (int n)
   // Returns the n-th NonlocalBarrier.
{

   if (nonlocalBarierList -> includes(n))
      return nonlocalBarierList -> at(n) ;
   else {
     _error2 ("giveNonlocalBarrier: undefined barrier (%d)",n);
   }
   return NULL;
}



/*
YieldCriteria*  Domain :: giveYieldCriteria (int n)
   // Returns the n-th yieldCriteria
   // yet.
{
   YieldCriteria* yieldCriteria ;

   if (yieldCriteriaList -> includes(n))
      yieldCriteria = (YieldCriteria*) yieldCriteriaList -> at(n) ;
   else {
     _errori ("giveYieldCriteria: No such cross section defined: ",n);
   }
   return yieldCriteria ;
}
*/


int  Domain :: giveNumberOfElements ()
   // Returns the number of elements the problem consists of.
{
   if (! numberOfElements)
     _error("giveNumberOfElements: Not defined");
   //      numberOfElements = this -> readNumberOf("Element") ;
   return numberOfElements ;
}




EngngModel*  Domain :: giveEngngModel ()
   // Returns the time integration algorithm. Creates it if it does not
   // exist yet.
{
 if (engineeringModel)
      return  engineeringModel;
   else {
  _error("giveEngngModel: Not defined");
   }
 
 return NULL;  
}


int  Domain :: instanciateYourself (DataReader* dr)
   // Creates all objects mentioned in the data file.

{
 const char *__keyword, *__proc = "initializeFrom"; // Required by IR_GIVE_FIELD macro
 IRResultType result;                               // Required by IR_GIVE_FIELD macro

   int i, num ;
   char name [MAX_NAME_LENGTH];
   int nnode,nelem,nmat,nload,nloadtimefunc,ncrossSections,nbarrier ;
   DofManager *node ;
   Element *elem ;
   Material *mat ;
   Load *load;
   LoadTimeFunction *ltf;
  CrossSection *crossSection;
  NonlocalBarrier *barrier;
  FILE *outputStream = this->giveEngngModel()->giveOutputStream();

 // read type of Domain to be solved
 InputRecord* ir = dr->giveInputRecord (DataReader::IR_domainRec, 1);
 __keyword = "domain"; result = ir->giveField(name, MAX_NAME_LENGTH, __keyword);
 if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, __keyword, ir, result);
 ir->finish();
 
#  ifdef VERBOSE
 VERBOSE_PRINT0("Instanciating domain ", this->number);
#  endif
 
 resolveDomainDofsDefaults (name);
 fprintf (outputStream,"Domain type: %s, default ndofs per node is %d, per side is %d\n\n\n",
      name, giveNumberOfDefaultNodeDofs (),giveNumberOfDefaultSideDofs() );
 
 // read otput manager record
 ir = dr->giveInputRecord (DataReader::IR_outManRec, 1);
 outputManager->initializeFrom (ir);
 ir->finish();

 // read domain description
 ir = dr->giveInputRecord (DataReader::IR_domainCompRec, 1);
 sscanf (ir->giveRecordAsString(),"%d %d %d %d %d %d",&nnode,&nelem,&ncrossSections,
      &nmat,&nload,&nloadtimefunc) ;
 numberOfElements = nelem;
 // read optional number of nonlocalBarriers
 nbarrier = 0; 
 __keyword = "nbarrier"; result = ir->giveOptionalField(nbarrier,  __keyword);

 // read nodes
 dofManagerList -> growTo(nnode) ;
 for (i=0; i < nnode ; i++) {
   ir = dr->giveInputRecord (DataReader::IR_dofmanRec, i+1);
   // read type of dofManager
   //__keyword = NULL; result = ir->giveField(name, MAX_NAME_LENGTH, __keyword);
   result = IR_GIVE_RECORD_KEYWORD_FIELD(ir, name, num, MAX_NAME_LENGTH);
   if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, "", ir, result);
   (node = (DofManager *) 
    (DofManager(i+1,this).ofType(name)))->initializeFrom(ir);

   // check number
   if ((num < 1) || (num > nnode)) 
     _error2 ("instanciateYourself: Invalid dofManager number (num=%d)", num);
   if (!dofManagerList->includes(num)) {
     dofManagerList->put(num,node) ;
   } else {
     _error2 ("instanciateYourself: Dofmanager entry already exist (num=%d)", num);
   }

   //dofManagerList->put(i+1,node) ;
   ir->finish();
 }
  
#  ifdef VERBOSE
  VERBOSE_PRINT0("Instanciated nodes & sides ",nnode)
#  endif

// read elements
   elementList -> growTo(nelem) ;
   for (i=0; i < nelem ; i++) { 
     ir = dr->giveInputRecord (DataReader::IR_elemRec, i+1);
     // read type of element
     //__keyword = NULL; result = ir->giveField(name, MAX_NAME_LENGTH, __keyword);
     result = IR_GIVE_RECORD_KEYWORD_FIELD(ir, name, num, MAX_NAME_LENGTH);
     if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, "", ir, result);
     (elem = (Element *) 
      (Element(i+1,this).ofType(name)))->initializeFrom(ir);

     // check number
     if ((num < 1) || (num > nelem)) 
       _error2 ("instanciateYourself: Invalid element number (num=%d)", num);
     if (!elementList->includes(num)) {
       elementList->put(num,elem) ;
     } else {
       _error2 ("instanciateYourself: element entry already exist (num=%d)", num);
     }

   //elementList->put(i+1,elem) ;
   ir->finish();
   }

#  ifdef VERBOSE
 VERBOSE_PRINT0("Instanciated elements ",nelem);
#  endif

// read cross sections
 crossSectionList -> growTo(ncrossSections);
   for (i=0; i < ncrossSections ; i++) {
     ir = dr->giveInputRecord (DataReader::IR_crosssectRec, i+1);
     //__keyword = NULL; result = ir->giveField(name, MAX_NAME_LENGTH, __keyword);
     result = IR_GIVE_RECORD_KEYWORD_FIELD(ir, name, num, MAX_NAME_LENGTH);
     if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, "", ir, result);
     (crossSection  = (CrossSection *) 
      (CrossSection(i+1,this).ofType(name)))->initializeFrom(ir);

     // check number
     if ((num < 1) || (num > ncrossSections)) 
       _error2 ("instanciateYourself: Invalid crossSection number (num=%d)", num);
     if (!crossSectionList->includes(num)) {
       crossSectionList->put(num,crossSection) ;
     } else {
       _error2 ("instanciateYourself: crossSection entry already exist (num=%d)", num);
     }
     //crossSectionList->put(i+1,crossSection) ;
     ir->finish();
   }   

#  ifdef VERBOSE
 VERBOSE_PRINT0 ("Instanciated cross sections ",ncrossSections) 
#  endif

  // read materials
  materialList -> growTo (nmat);
  for (i=0; i < nmat ; i++) { 
    ir = dr->giveInputRecord (DataReader::IR_matRec, i+1);
    // read type of material
    //__keyword = NULL; result = ir->giveField(name, MAX_NAME_LENGTH, __keyword);
    result = IR_GIVE_RECORD_KEYWORD_FIELD(ir, name, num, MAX_NAME_LENGTH);
    if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, "", ir, result);
    (mat  = (Material *) 
     (Material(i+1,this).ofType(name)))->initializeFrom(ir);

    // check number
    if ((num < 1) || (num > nmat)) 
      _error2 ("instanciateYourself: Invalid material number (num=%d)", num);
    if (!materialList->includes(num)) {
      materialList->put(num, mat) ;
    } else {
      _error2 ("instanciateYourself: material entry already exist (num=%d)", num);
    }
    //materialList->put(i+1,mat) ;
    ir->finish();
  }   

#  ifdef VERBOSE
 VERBOSE_PRINT0 ("Instanciated materials ",nmat) 
#  endif

  if (nbarrier) {
    // read barriers
    nonlocalBarierList -> growTo (nbarrier);
    for (i=0; i < nbarrier ; i++)
      { 
        ir = dr->giveInputRecord (DataReader::IR_nlocBarRec, i+1);
        // read type of load
        //__keyword = NULL; result = ir->giveField(name, MAX_NAME_LENGTH, __keyword);
        result = IR_GIVE_RECORD_KEYWORD_FIELD(ir, name, num, MAX_NAME_LENGTH);
        if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, "", ir, result);
        barrier = CreateUsrDefNonlocalBarrierOfType (name, i+1, this);
        barrier->initializeFrom(ir);

        // check number
        if ((num < 1) || (num > nbarrier)) 
          _error2 ("instanciateYourself: Invalid barrier number (num=%d)", num);
        if (!nonlocalBarierList->includes(num)) {
          nonlocalBarierList->put(num, barrier) ;
        } else {
          _error2 ("instanciateYourself: barrier entry already exist (num=%d)", num);
        }
        //nonlocalBarierList->put(i+1,barrier) ;
        ir->finish();
      }
#  ifdef VERBOSE
    VERBOSE_PRINT0 ("Instanciated barriers ",nbarrier) ;
#  endif
   
  }


// read loads
 loadList -> growTo (nload);
 for (i=0; i < nload ; i++) {  
   ir = dr->giveInputRecord (DataReader::IR_bcicRec, i+1);
   // read type of load
   //__keyword = NULL; result = ir->giveField(name, MAX_NAME_LENGTH, __keyword);
   result = IR_GIVE_RECORD_KEYWORD_FIELD(ir, name, num, MAX_NAME_LENGTH);
   if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, "", ir, result);
   (load = (Load *) 
    (GeneralBoundaryCondition(i+1,this).ofType(name)))->initializeFrom(ir);

   // check number
   if ((num < 1) || (num > nload)) 
     _error2 ("instanciateYourself: Invalid boundary condition number (num=%d)", num);
   if (!loadList->includes(num)) {
     loadList->put(num, load) ;
   } else {
     _error2 ("instanciateYourself: boundary condition entry already exist (num=%d)", num);
   }
   //loadList->put(i+1,load) ;
   ir->finish();
 }


#  ifdef VERBOSE
 VERBOSE_PRINT0 ("Instanciated loads ",nload)
#  endif


 // read load time functions
 loadTimeFunctionList -> growTo (nloadtimefunc);
 for (i=0; i < nloadtimefunc ; i++) {  
   ir = dr->giveInputRecord (DataReader::IR_ltfRec, i+1);
   // read type of ltf
   //__keyword = NULL; result = ir->giveField(name, MAX_NAME_LENGTH, __keyword);
   result = IR_GIVE_RECORD_KEYWORD_FIELD(ir, name, num, MAX_NAME_LENGTH);
   if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, "", ir, result);
   (ltf  = (LoadTimeFunction *) 
    (LoadTimeFunction(i+1,this).ofType(name)))->initializeFrom(ir);

   // check number
   if ((num < 1) || (num > nloadtimefunc)) 
     _error2 ("instanciateYourself: Invalid LoadTimeFunction number (num=%d)", num);
   if (!loadTimeFunctionList->includes(num)) {
     loadTimeFunctionList->put(num, ltf) ;
   } else {
     _error2 ("instanciateYourself: LoadTimeFunction entry already exist (num=%d)", num);
   }
   //loadTimeFunctionList->put(i+1,ltf) ;
   ir->finish();
 }   

#  ifdef VERBOSE
 VERBOSE_PRINT0 ("Instanciated load-time fncts ",nloadtimefunc)
#  endif

  return 1;
}


void  Domain :: error (char* file, int line, char *format, ...) 
{
  char buffer[MAX_ERROR_MSG_LENGTH];
	va_list args;

	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);
  
  __OOFEM_ERROR3 (file, line, "Class: Domain, number: %d\n%s",number,buffer);
}


void  Domain :: warning (char* file, int line, char *format, ...) 
{
  char buffer[MAX_ERROR_MSG_LENGTH];
	va_list args;

	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);
  
  __OOFEM_WARNING3 (file, line, "Class: Domain, number: %d\n%s",number,buffer);
}


const IntArray&
Domain :: giveDefaultNodeDofIDArry ()
{
// returns default DofID array, defining physical meaning of partucular DOFs
// in Node Dof collection
 if (this->defaultNodeDofIDArry.giveSize()) {
  return defaultNodeDofIDArry;
 }


  if(dType == _2dPlaneStressRotMode) {
  defaultNodeDofIDArry.resize(3);
    defaultNodeDofIDArry.at(1)=D_u; defaultNodeDofIDArry.at(2)=D_v;defaultNodeDofIDArry.at(3)=R_w;
 }
  else if(dType == _2dPlaneStressMode) {
  defaultNodeDofIDArry.resize (2);
    defaultNodeDofIDArry.at(1)=D_u; defaultNodeDofIDArry.at(2)=D_v;
 }
  else if(dType == _PlaneStrainMode) {
  defaultNodeDofIDArry.resize (2);
    defaultNodeDofIDArry.at(1)=D_u; defaultNodeDofIDArry.at(2)=D_v;
 }
  else if  (dType == _3dMode) {
  defaultNodeDofIDArry.resize (3);
    defaultNodeDofIDArry.at(1)=D_u; defaultNodeDofIDArry.at(2)=D_v; defaultNodeDofIDArry.at(3)=D_w;
 }
 else if (dType == _3dAxisymmMode) {
  defaultNodeDofIDArry.resize (3);
    defaultNodeDofIDArry.at(1)=D_u; defaultNodeDofIDArry.at(2)=D_v; defaultNodeDofIDArry.at(3)=R_w;
 }
  else if  (dType == _2dMindlinPlateMode) {
  defaultNodeDofIDArry.resize (3);
    defaultNodeDofIDArry.at(1)=D_w; defaultNodeDofIDArry.at(2)=R_u; defaultNodeDofIDArry.at(3)=R_v;
 }
 else if ( dType == _3dShellMode) {
  defaultNodeDofIDArry.resize (6);
    defaultNodeDofIDArry.at(1)=D_u; defaultNodeDofIDArry.at(2)=D_v; defaultNodeDofIDArry.at(3)=D_w;
  defaultNodeDofIDArry.at(4)=R_u; defaultNodeDofIDArry.at(5)=R_v; defaultNodeDofIDArry.at(6)=R_w;
 }
  else if  (dType == _2dTrussMode) {
  defaultNodeDofIDArry.resize (2);
    defaultNodeDofIDArry.at(1)=D_u; defaultNodeDofIDArry.at(2)=D_w;
 }
  else if  (dType == _1dTrussMode) {
  defaultNodeDofIDArry.resize(1);
    defaultNodeDofIDArry.at(1)=D_u; 
 }
  else if  (dType == _2dBeamMode) {
  defaultNodeDofIDArry.resize (3);
    defaultNodeDofIDArry.at(1)=D_u; defaultNodeDofIDArry.at(2)=D_w; defaultNodeDofIDArry.at(3)=R_v;
 }
  else if  (dType == _HeatTransferMode) {
  defaultNodeDofIDArry.resize (1);
    defaultNodeDofIDArry.at(1)=T_f;
 }  
  else if  (dType == _HeatMass1Mode) {
  defaultNodeDofIDArry.resize (2);
    defaultNodeDofIDArry.at(1)=T_f;
    defaultNodeDofIDArry.at(2)=C_1;
 }  
  else {
    _error("giveDefaultNodeDofIDArry : unknown domainType");
  }
 return defaultNodeDofIDArry;
}



int Domain ::  giveNumberOfDefaultNodeDofs () 
//
// returns default number of dofs for one node
// this number depend on type of problem (2dplane-stress, 3d truss, 3d, 2d beam etc.)
// returns member data  numberOfDefaultDofsPerNode.
// numberOfDefaultDofsPerNode is initialized in initiazeFrom subroutine.
// 
{
  if ( numberOfDefaultDofsPerNode == -1) {
    OOFEM_LOG_WARNING("Domain ::  giveNumberOfDefaultNodeDofs : Number of Default Dofs per Node is not specified, using default 6 instead\a\n");
    return (numberOfDefaultDofsPerNode = 6);
  } else return  numberOfDefaultDofsPerNode ;
}


const IntArray&
Domain :: giveDefaultSideDofIDArry ()
{
// returns default DofID array, defining physical meaning of partucular DOFs
// in side Dof collection

  // IntArray* answer;

/*
  if(dType == _2dPlaneStressRotMode) {
  answer = new IntArray (3);
    answer->at(1)=D_u; answer->at(2)=D_v;answer->at(3)=R_w;
 }
  else if(dType == _2dPlaneStressMode) {
  answer = new IntArray (2);
    answer->at(1)=D_u; answer->at(2)=D_v;
 }
  else if  (dType == _3dMode) {
  answer = new IntArray (3);
    answer->at(1)=D_u; answer->at(2)=D_v; answer->at(3)=D_w;
 }
 else if (dType == _3dAxisymmMode) {
  answer = new IntArray (3);
    answer->at(1)=D_u; answer->at(2)=D_v; answer->at(3)=R_w;
 }
  else if  (dType == _2dMindlinPlateMode) {
  answer = new IntArray (3);
    answer->at(1)=D_w; answer->at(2)=R_u; answer->at(3)=R_v;
 }
 else if ( dType == _3dShellMode) {
  answer = new IntArray (5);
    answer->at(1)=D_u; answer->at(2)=D_v; answer->at(3)=D_w;
  answer->at(4)=R_u; answer->at(5)=R_v;
 }
  else if  (dType == _2dTrussMode) {
  answer = new IntArray (2);
    answer->at(1)=D_u; answer->at(2)=D_w;
 }
  else if  (dType == _1dTrussMode) {
  answer = new IntArray (1);
    answer->at(1)=D_u; 
 }
  else if  (dType == _2dBeamMode) {
  answer = new IntArray (3);
    answer->at(1)=D_u; answer->at(2)=D_w; answer->at(3)=R_v;
 }
  else if  (dType == _2dHeatMode) {
  answer = new IntArray (1);
    answer->at(1)=T_f;
 }  
  else {
    _error("Domain : Domain type name of unknown type\a\n");
    return NULL;
  }
 return answer;
 */

  _error("giveDefaultSideDofIDArry : unknown domainType");
  defaultSideDofIDArry.resize(0);
  return defaultSideDofIDArry;
}



int Domain ::  giveNumberOfDefaultSideDofs () 
//
// returns default number of dofs for one side
// this number depend on type of problem (2dplane-stress, 3d truss, 3d, 2d beam etc.)
// returns member data  numberOfDefaultDofsPerNode.
// numberOfDefaultDofsPerNode is initialized in initializeFrom subroutine.
// 
{
  if ( numberOfDefaultDofsPerSide == -1) {
    _warning("giveNumberOfDefaultSideDofs: Number of Default Dofs per Side is not specified, using default 0 instead");
    return (numberOfDefaultDofsPerSide = 0);
  } else return  numberOfDefaultDofsPerSide ;
}

 



void Domain ::  resolveDomainDofsDefaults(char* typeName)
//
// resolves default number of dofs per node according to domain type name.
// and also resolves default dof mask according to domain type.
//
{
 numberOfDefaultDofsPerSide = 0;

  if(!strncasecmp(typeName,"2dplanestressrot",16)) {
    dType = _2dPlaneStressRotMode; 
  numberOfDefaultDofsPerNode = 3;
 }
  else if(!strncasecmp(typeName,"2dplanestress",12)) {
    dType = _2dPlaneStressMode; 
  numberOfDefaultDofsPerNode = 2;
 }
  else if(!strncasecmp(typeName,"planestrain",11)) {
    dType = _PlaneStrainMode; 
  numberOfDefaultDofsPerNode = 2;
 }
 else if (! strncasecmp(typeName,"3daxisymm",9)) {
    dType = _3dAxisymmMode;
  numberOfDefaultDofsPerNode = 3;
 }
  else if  (! strncasecmp(typeName,"2dmindlinplate",14)) {
    dType = _2dMindlinPlateMode;
  numberOfDefaultDofsPerNode = 3;
 }
 else if (! strncasecmp(typeName,"3dshell",7)) {
    dType = _3dShellMode;
  numberOfDefaultDofsPerNode = 6;
 }
  else if  (! strncasecmp(typeName,"2dtruss",7)){
    dType = _2dTrussMode; 
  numberOfDefaultDofsPerNode = 2;
 }
  else if  (! strncasecmp(typeName,"1dtruss",7)){
    dType = _1dTrussMode; 
  numberOfDefaultDofsPerNode = 1;
 }
  else if  (! strncasecmp(typeName,"2dbeam",6)) {
    dType = _2dBeamMode;  
  numberOfDefaultDofsPerNode = 3;
 }
  else if  (! strncasecmp(typeName,"3d",2))     {
    dType = _3dMode;
  numberOfDefaultDofsPerNode = 3;
 }
  else if  (! strncasecmp(typeName,"heattransfer",11)) {
    dType = _HeatTransferMode;
  numberOfDefaultDofsPerNode = 1;
 }  
  else if  (! strncasecmp(typeName,"hema1",5)) {
    dType = _HeatMass1Mode;
  numberOfDefaultDofsPerNode = 2;
 }  
  else { 
    _error("resolveDomainDofsDefaults : unknown domainType");
    return;
  }
}
    

#ifdef __OOFEG

void Domain :: drawYourself (oofegGraphicContext& context) 
//
// shows graphics representation of domain, respecting mode
//
{
 
 OGC_PlotModeType plotMode = context.giveIntVarPlotMode();
 if ((plotMode == OGC_nodeAnnotation) || (plotMode == OGC_nodeGeometry) || (plotMode == OGC_appliedPrimaryBc))
    this->drawNodes (context);
 else 
  this->drawElements (context);
}

void Domain :: drawElements (oofegGraphicContext& context) {
//
// steps through element array and calls element(i)->show(mode,this);
//
  for (int i=1; i <= numberOfElements ; i++)
    {  
      this->giveElement(i)->drawYourself(context);
    }
}

void  Domain :: drawNodes (oofegGraphicContext& context) {
//
// steps through element array and calls element(i)->show(mode,this);
//
  int nnodes = this->giveNumberOfDofManagers();
  for (int i=1; i <= nnodes ; i++)
    {  
      this->giveDofManager(i)->drawYourself(context);
    }
}

#endif


NodalRecoveryModel* 
Domain :: giveSmoother ()
{
 return this->smoother;
}

void
Domain :: setSmoother (NodalRecoveryModel* smoother, int destroyOld)
{
 if (destroyOld && this->smoother) delete this->smoother;
 this->smoother = smoother;
}




ConnectivityTable * Domain :: giveConnectivityTable () 
// 
// return connectivity Table - if no defined - creates new one
//
{
  if (connectivityTable == NULL) connectivityTable = new ConnectivityTable(this);
  return connectivityTable;
}


SpatialLocalizer * Domain :: giveSpatialLocalizer () 
// 
// return connectivity Table - if no defined - creates new one
//
{

//  if (spatialLocalizer == NULL) spatialLocalizer = new DummySpatialLocalizer(1, this);
 if (spatialLocalizer == NULL) spatialLocalizer = new OctreeSpatialLocalizer(1, this);
  return spatialLocalizer;
}




int Domain ::  giveCorrespondingCoordinateIndex (int idof)
//
// find corresponding coordinate axis to idof
// if no - coordinate axis corespond to idof returns 0;
// 
// if idof corresponds to displacement in direction of axis i then finction returns i
// otherwise 0;
//
{
  switch (dType) {
  case _2dBeamMode:
    if (idof == 1) return 1; else if(idof == 2) return 3;
    return 0;

  case _2dPlaneStressMode:
 case _PlaneStrainMode:
    if (idof == 1) return 1; else if(idof == 2) return 2;
    return 0;

  case _2dPlaneStressRotMode:
    if (idof == 1) return 1; else if(idof == 2) return 2;
    return 0;

  case _2dTrussMode:
    if (idof == 1) return 1; else if(idof == 2) return 3;
    return 0;

  case _1dTrussMode:
    if (idof == 1) return 1; 
    return 0;

  case _2dMindlinPlateMode:
    if (idof == 1) return 3; 
    return 0;

 case _3dMode:
  if (idof == 1) return 1; else if (idof == 2) return 2;
  else if (idof == 3) return 3;
  return 0;

 case _3dAxisymmMode:
  if (idof == 1) return 1; else if (idof == 2) return 2;
  return 0;

 case _3dShellMode:
  if (idof == 1) return 1; else if (idof == 2) return 2;
  else if (idof == 3) return 3;
  return 0;


  
  default:
    _error ("giveCorrespondingCoordinateIndex : unsupported domain type");
  }
return 0;
}

/*
Domain :: giveCorrespondingDofID (int idof)
{
// returns corresponding DofId to idof-th dof in node
// respecting current domain mode.
// if no corresponding dofID exists returns (Err_dof = 0)
//
 
 switch (dType) {
 case _2dBeamMode:
  if     (idof == 1) return D_u; 
  else if(idof == 2) return D_w;
  else if(idof == 3) return R_v;
  break ;
 case _2dPlaneStressMode:
  if     (idof == 1) return D_u;
  else if(idof == 2) return D_v;
  break;
 case _2dTrussMode:
  if     (idof == 1) return D_u;
  else if(idof == 2) return D_v;
  break;
 case _1dTrussMode:
  if     (idof == 1) return D_u;
  break;
 case _2dMindlinPlateMode:
  if     (idof == 1) return D_w;
  else if(idof == 2) return R_u;
  else if(idof == 3) return R_v;
  break;
 case _3dMode:
  if     (idof == 1) return D_u;
  else if(idof == 2) return D_v;
  else if(idof == 3) return D_w;
  break;
 case _2dHeatMode:
   if     (idof == 1) return T_f;
   break;
 default:
  _error ("giveCorrespondingDofID : udefined iDof for selected domainType");
 }
 return Err_dof;
 
}
*/

int
Domain :: checkConsistency ()
//
// checks internal consistency
// 
// many parameters are checked at run-time during computation
//
// this function transverse tree of all objects and invokes
// checkConsistency on this objects
// currently this function checks noly consistency
// of internal object structures, mainly whether referenced other objects
// are having required support
// 
{
  int i, result=1;
  int nnode, nelem;
  
  nnode = this-> giveNumberOfDofManagers ();
  nelem = this-> giveNumberOfElements();

  for (i=1; i <= nnode ; i++)
  result &= this->giveDofManager(i)->checkConsistency();

  for (i=1; i <= nelem ; i++)
  result &= this->giveElement(i)->checkConsistency();

  return result;
}

//


