/* $Header: /home/cvs/bp/oofem/oofemlib/src/element.C,v 1.19.4.1 2004/04/05 15:19:43 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/


//   file ELEMENT.CC

#include "element.h"
//#include "planstrss.h"
//#include "qplanstrss.h"
//#include "trplanstrss.h"
//#include "lspace.h"
//#include "qspace.h"
//#include "cct.h"
//#include "ltrspace.h"
//#include "truss2d.h"
//#include "libeam2d.h"
//#include "trplanrot.h"
//#include "qtrplstr.h"
//#include "axisymm3d.h"
//#include "q4axisymm.h"
//#include "l4axisymm.h"
//#include "rershell.h"
//#include "ltrelemppde.h"

#include "domain.h"
#include "timestep.h"
#include "node.h"
#include "dof.h"
#include "boundary.h"
#include "material.h"
#include "crosssection.h"
#include "bodyload.h"
#include "gausspnt.h"
#include "integrationrule.h"
#include "intarray.h"
#include "flotarry.h"
#include "flotmtrx.h"
//#include "linsyst.h"
#include "skyline.h"
#include "debug.h"
#include "verbose.h"
#include "cltypes.h"
#include "usrdefsub.h"
#include "elementside.h"
#ifndef __MAKEDEPEND
#include <stdlib.h>
#include <stdio.h>
#endif

//

Element :: Element (int n, Domain* aDomain)
  : FEMComponent (n, aDomain), nodeArray(), sideArray(), bodyLoadArray(), boundaryLoadArray()
   // Constructor. Creates an element with number n, belonging to aDomain.
{
   material           = 0    ;
   numberOfNodes      = 0    ;
  numberOfSides      = 0    ;
  numberOfIntegrationRules = 0;
   locationArray      = NULL ;
   integrationRulesArray  = NULL ;
}


Element :: ~Element ()
   // Destructor.
{
   int i ;

   delete locationArray ;
   if (integrationRulesArray) {
     for (i=0 ; i<numberOfIntegrationRules ; i++)
       delete integrationRulesArray[i] ;
     //      delete [numberOfGaussPoints] gaussPointArray ;}
     delete integrationRulesArray ;}
}

void
Element :: computeVectorOf (UnknownType type, ValueModeType u, TimeStep* stepN, FloatArray& answer)
   // Forms the vector containing the values of the unknown 'u' (e.g., the
   // Total value) of the dofs of the receiver's nodes (in nodal cs).
   // Dofs cointaining expected unknowns (of expected type) are determined 
   // using this->GiveNodeDofIDMask function
{
   // FloatArray *answer ;
   int        i,j,k,nDofs,size ;
  IntArray   elementNodeMask;
  FloatArray vec;
   //answer = new FloatArray(size = this->computeNumberOfDofs()) ;
  answer.resize (size = this->computeGlobalNumberOfDofs());

   k      = 0 ;
   for (i=1 ; i<=numberOfNodes ; i++) {
   this ->  giveNodeDofIDMask (i, elementNodeMask);
   this->giveNode(i)->giveUnknownVector (vec, elementNodeMask, type, u, stepN);
   nDofs = vec.giveSize();
   for (j=1 ; j<=nDofs ; j++)
    answer.at(++k) = vec.at(j);
   //delete elementNodeMask;
   //delete dofMask;
  }
   for (i=1 ; i<=numberOfSides ; i++) {
   this ->  giveSideDofIDMask (i, elementNodeMask);
   this->giveSide(i)->giveUnknownVector (vec, elementNodeMask, type, u, stepN);
   nDofs = vec.giveSize() ;
   for (j=1 ; j<=nDofs ; j++)
     answer.at(++k) = vec.at(j);
   // delete elementNodeMask;
   // delete dofMask;
  }

  if (size != k) _error ("computeVectorOf: Unknown vector and location array size mismatch");
 
  return  ;
 }

void
Element :: computeVectorOf (PrimaryField& field, ValueModeType u, TimeStep* stepN, FloatArray& answer)
   // Forms the vector containing the values of the unknown 'u' (e.g., the
   // Total value) of the dofs of the receiver's nodes (in nodal cs).
   // Dofs cointaining expected unknowns (of expected type) are determined 
   // using this->GiveNodeDofIDMask function
{
   int        i,j,k,nDofs,size ;
  IntArray   elementNodeMask;
  FloatArray vec;
  answer.resize (size = this->computeGlobalNumberOfDofs());

   k      = 0 ;
   for (i=1 ; i<=numberOfNodes ; i++) {
   this ->  giveNodeDofIDMask (i, elementNodeMask);
   this->giveNode(i)->giveUnknownVector (vec, elementNodeMask, field, u, stepN);
   nDofs = vec.giveSize();
   for (j=1 ; j<=nDofs ; j++)
    answer.at(++k) = vec.at(j);
  }
   for (i=1 ; i<=numberOfSides ; i++) {
   this ->  giveSideDofIDMask (i, elementNodeMask);
   this->giveSide(i)->giveUnknownVector (vec, elementNodeMask, field, u, stepN);
   nDofs = vec.giveSize() ;
   for (j=1 ; j<=nDofs ; j++)
     answer.at(++k) = vec.at(j);
  }
  
  if (size != k) _error ("computeVectorOf: Unknown vector and location array size mismatch");
  
  return  ;
 }

void 
Element :: computeVectorOfPrescribed (ValueModeType mode, TimeStep* stepN,
                   FloatArray& answer)
   // Forms the vector containing the prescribed values of the unknown 'u'
   // (e.g., the prescribed displacement) of the dofs of the receiver's
   // nodes. Puts 0 at each free dof.
{
   // FloatArray *answer ;
   int        i,j,k,size,nDofs ;
  IntArray   elementNodeMask, dofMask;
  FloatArray vec;

  // answer = new FloatArray(this->computeNumberOfDofs()) ;
  answer.resize (size = this->computeGlobalNumberOfDofs());

   k      = 0 ;
  for (i=1 ; i<=numberOfNodes ; i++) {
   this ->  giveNodeDofIDMask (i, elementNodeMask);
   this->giveNode(i)->givePrescribedUnknownVector (vec, elementNodeMask, mode, stepN);
   nDofs = vec.giveSize();
   for (j=1 ; j<=nDofs ; j++)
    answer.at(++k) = vec.at(j);
   //delete elementNodeMask;
   //delete dofMask;
  }
   for (i=1 ; i<=numberOfSides ; i++) {
   this ->  giveSideDofIDMask (i, elementNodeMask);
   this->giveSide(i)->givePrescribedUnknownVector (vec, elementNodeMask, mode, stepN);
   nDofs = vec.giveSize() ;
   for (j=1 ; j<=nDofs ; j++)
     answer.at(++k) = vec.at(j);
   // delete elementNodeMask;
   // delete dofMask;
  }

  if (size != k) _error ("computeVectorOf: Unknown vector and location array size mismatch");
  
  return  ;
 }


int
Element :: computeGlobalNumberOfDofs () 
{
 if (this->locationArray) {
  return this->locationArray->giveSize();
 } else {
  IntArray loc;
  this-> giveLocationArray (loc);
  return loc.giveSize();
 }
}

IntArray*  Element :: giveBodyLoadArray ()
   // Returns the array which contains the number of every body load that act
   // on the receiver.
{
   return &bodyLoadArray ;
}


IntArray*  Element :: giveBoundaryLoadArray ()
   // Returns the array which contains the number of every body load that act
   // on the receiver.
{
   return &boundaryLoadArray ;
}


void  Element :: giveLocationArray (IntArray& locationArray) const
   // Returns the location array of the receiver. This array is obtained by
   // simply appending the location array of every node of the receiver.
{
 IntArray  nodeDofIDMask ;
 IntArray  nodalArray;
 int       i ;

 if (this->locationArray) {
  locationArray = *this->locationArray;
  return;
 } else {

  locationArray.resize (0);
  for (i=1 ; i<=numberOfNodes ; i++) {
   this->giveNodeDofIDMask (i, nodeDofIDMask);
   this -> giveNode(i) -> giveLocationArray(nodeDofIDMask, nodalArray) ;
   locationArray.followedBy(nodalArray) ;
   // delete nodeDofIDMask;
  }
  for (i=1 ; i<=numberOfSides ; i++) {
   this->giveSideDofIDMask (i, nodeDofIDMask);
   this -> giveSide(i) -> giveLocationArray(nodeDofIDMask, nodalArray) ;
   locationArray.followedBy(nodalArray) ;
   // delete nodeDofIDMask;
  }
 }
 return ;
}

void  Element :: givePrescribedLocationArray (IntArray& locationArray) const
   // Returns the location array of the receiver. This array is obtained by
   // simply appending the location array of every node of the receiver.
{
 IntArray  nodeDofIDMask ;
 IntArray  nodalArray;
 int       i ;

 locationArray.resize (0);
 for (i=1 ; i<=numberOfNodes ; i++) {
  this->giveNodeDofIDMask (i, nodeDofIDMask);
  this -> giveNode(i) -> givePrescribedLocationArray(nodeDofIDMask, nodalArray) ;
  locationArray.followedBy(nodalArray) ;
 }
 for (i=1 ; i<=numberOfSides ; i++) {
  this->giveSideDofIDMask (i, nodeDofIDMask);
  this -> giveSide(i) -> givePrescribedLocationArray(nodeDofIDMask, nodalArray) ;
  locationArray.followedBy(nodalArray) ;
 }

 return ;
}

void 
Element :: invalidateLocationArray ()
{
 // invalitaes current location array in receiver
 // next call of giveLocationArray() will asemble
 // new location array
 // used mainly for model supporting dynamic changes of
 // static system


 // force assembling
 // of new location array
 if (locationArray) delete locationArray;
 locationArray = NULL;
}

Material*  Element :: giveMaterial ()
   // Returns the material of the receiver.
{
   if (! material)
     // material = this -> readInteger("mat") ;
     _error ("giveMaterial: material not defined");

   return  domain -> giveMaterial(material) ;
}


CrossSection*  Element :: giveCrossSection ()
   // Returns the crossSection of the receiver.
{
   if (! crossSection)
     _error ("giveCrossSection: crossSection not defined");

   return  domain -> giveCrossSection (crossSection) ;
}

int
Element::giveRegionNumber () 
{
 return this->giveCrossSection()->giveNumber();
}


Node*  Element :: giveNode (int i) const
   // Returns the i-th node of the receiver.
{
   int  n ;
  if ((i <= 0) || (i > nodeArray.giveSize()))
    _error ("giveNode: Node is not defined");

   n = nodeArray.at(i) ;
   return  domain -> giveNode(n) ;
}


ElementSide*  Element :: giveSide (int i) const
   // Returns the i-th side of the receiver.
{
   int  n ;

  if ((i <= 0) || (i > sideArray.giveSize()))
   _error ("giveNode: Side is not defined");
   n = sideArray.at(i) ;
   return  domain -> giveSide(n) ;
}





void
Element ::  giveCharacteristicMatrix (FloatMatrix &answer, 
                   CharType mtrx, TimeStep *tStep) 
// 
// returns characteristics matrix of receiver accordind to mtrx
//
{
  _error("giveCharacteristicMatrix: Unknown Type of characteristic mtrx.");
  return ;
}



void
Element ::  giveCharacteristicVector (FloatArray& answer, CharType type, ValueModeType mode, TimeStep *tStep) 
// 
// returns characteristics vector of receiver accordind to mtrx
//
{
  _error("giveCharacteristicVector: Unknown Type of characteristic mtrx.");
  return ;
}



double Element ::  giveCharacteristicValue  (CharType mtrx, TimeStep *tStep) 
// 
// returns characteristics value of receiver accordind to CharType
//
{
  _error("giveCharacteristicValue: Unknown Type of characteristic mtrx.");
  return 0.;
}

IRResultType
Element :: initializeFrom (InputRecord* ir)
{
 const char *__keyword, *__proc = "initializeFrom"; // Required by IR_GIVE_FIELD macro
 IRResultType result;                               // Required by IR_GIVE_FIELD macro
 
 
#  ifdef VERBOSE
// VERBOSE_PRINT1("Instanciating element ",number);
#  endif
 IR_GIVE_FIELD (ir, material, "mat"); // Macro

 IR_GIVE_FIELD (ir, crossSection, "crosssect"); // Macro

 IR_GIVE_FIELD (ir, nodeArray, "nodes"); // Macro

 sideArray.resize(0);
 IR_GIVE_OPTIONAL_FIELD (ir, sideArray, "sides"); // Macro
 
 bodyLoadArray.resize(0);
 IR_GIVE_OPTIONAL_FIELD (ir, bodyLoadArray, "bodyloads"); // Macro

 boundaryLoadArray.resize(0);
 IR_GIVE_OPTIONAL_FIELD (ir, boundaryLoadArray, "boundaryloads"); // Macro

#ifdef __PARALLEL_MODE
 globalNumber = 0;
 IR_GIVE_OPTIONAL_FIELD (ir, globalNumber, "globnum"); // Macro
 partitions.resize(0);
 IR_GIVE_OPTIONAL_FIELD (ir, partitions, "partitions"); // Macro
 // if (hasString (initString, "shared")) parallel_mode = Element_shared;
 if (ir->hasField ("remote")) parallel_mode = Element_remote;
 else parallel_mode = Element_local;

#endif

 return IRRT_OK;
}

 
Element*  Element :: ofType (char* aClass)
   // Returns a new element, which has the same number than the receiver,
   // but belongs to aClass (PlaneStrain, or Truss2D,..).
{
   Element* newElement ;

/*   if (! strncasecmp(aClass,"planestress2d",12))
     newElement = new PlaneStress2d(number,domain) ;
   else if (! strncasecmp(aClass,"qplanestress2d",12))
     newElement = new QPlaneStress2d(number,domain) ; 
   else if (! strncasecmp(aClass,"trplanestress2d",12))
     newElement = new TrPlaneStress2d(number,domain) ;
   else if (! strncasecmp(aClass,"qtrplstr",8))
     newElement = new QTrPlaneStress2d(number,domain) ;
   else if (! strncasecmp(aClass,"trplanestrrot",12))
     newElement = new TrPlaneStrRot(number,domain) ;
   else if (! strncasecmp(aClass,"axisymm3d",9))
     newElement = new Axisymm3d (number,domain) ;
   else if (! strncasecmp(aClass,"q4axisymm",9))
     newElement = new Q4Axisymm (number,domain) ;
   else if (! strncasecmp(aClass,"l4axisymm",9))
     newElement = new L4Axisymm (number,domain) ;
   else if (! strncasecmp(aClass,"lspace",6))
     newElement = new LSpace(number,domain) ; 
   else if (! strncasecmp(aClass,"qspace",6))
     newElement = new QSpace(number,domain) ; 
   else if (! strncasecmp(aClass,"cctplate",8))
     newElement = new CCTPlate(number,domain) ; 
   else if (! strncasecmp(aClass,"ltrspace",8))
     newElement = new LTRSpace(number,domain) ; 
   else if (! strncasecmp(aClass,"truss2d",7))
     newElement = new Truss2d(number,domain) ; 
   else if (! strncasecmp(aClass,"libeam2d",8))
     newElement = new LIBeam2d(number,domain) ;
   else if (! strncasecmp(aClass,"rershell",8))
     newElement = new RerShell(number,domain) ;
   else if (! strncasecmp(aClass,"ltrelemppde",11))
     newElement = new LTrElementPPDE(number,domain) ;
*/ if (0) {}
   else {
   // last resort - call aditional user defined subroutine
   newElement = ::CreateUsrDefElementOfType (aClass,number,domain);
   if (newElement == NULL) {
     _error2 ("ofType : unknown element type (%s)",aClass) ;
   }
  }
   return newElement ;
}


void  Element :: printOutputAt (FILE* file, TimeStep* stepN)
   // Performs end-of-step operations.
{
   int         i ;

#ifdef __PARALLEL_MODE
  fprintf (file,"element %d [%8d] :\n",this->giveNumber(), this->giveGlobalNumber()) ;
#else
   fprintf (file,"element %d :\n",number) ;
#endif

   for (i=0 ; i < numberOfIntegrationRules ; i++) 
      integrationRulesArray[i]->printOutputAt(file,stepN);
}

/*
Element*  Element :: typed ()
   // Returns a new element, which has the same number than the receiver,
   // but is typed (PlaneStrain, or Truss2D,..).
{
   Element* newElement ;
   char     type[32] ;

   this -> readString("class",type) ;
   newElement = this -> ofType(type) ;

   return newElement ;
}
*/

void  Element :: updateYourself (TimeStep* tStep)
   // Updates the receiver at end of step.
{
 int i ;

#  ifdef VERBOSE
// VERBOSE_PRINT1("Updating Element ",number)
#  endif
 for (i=0 ; i < numberOfIntegrationRules ; i++)
  integrationRulesArray[i] -> updateYourself(tStep) ;
}

void  Element ::initForNewStep ()
   // initializes receiver to new time step or can be used
   // if current time step must be restarted
   // 
   // call material->initGpForNewStep() for all GPs.
   //
{
 int i;
 
 for (i=0 ; i < numberOfIntegrationRules ; i++)
  integrationRulesArray[i] -> initForNewStep ();
}



contextIOResultType Element :: saveContext (FILE* stream, void *obj)
//
// saves full element context (saves state variables, that completely describe
// current state)
//
{
  contextIOResultType iores;
 int         i ;

  if ((iores = FEMComponent::saveContext(stream,obj)) != CIO_OK) THROW_CIOERR(iores);
  for (i=0 ; i < numberOfIntegrationRules ; i++) {
    if ((iores = integrationRulesArray[i]->saveContext(stream,obj)) != CIO_OK) THROW_CIOERR(iores);
  }

  return CIO_OK;
}



contextIOResultType Element :: restoreContext (FILE* stream, void *obj)
//
// restores full element context (saves state variables, that completely describe
// current state)
//
{
  contextIOResultType iores;
 int         i ;

  if ((iores = FEMComponent::restoreContext(stream,obj)) != CIO_OK) THROW_CIOERR(iores);
  for (i=0 ; i < numberOfIntegrationRules ; i++) {
    if ((iores = integrationRulesArray[i]->restoreContext(stream,obj)) != CIO_OK) THROW_CIOERR(iores);
  }

  return CIO_OK;
}



double
Element :: giveLenghtInDir (const FloatArray& normalToCrackPlane)
//
// returns receivers projection length (for some material models)
// to direction given by normalToCrackPlane;
//
{
 
 FloatArray *coords;
 double maxDis, minDis, dis;
 double *p1,*p2;
 int i,nsize;

 p1 = normalToCrackPlane.givePointer();
 coords = this->giveNode(1)->giveCoordinates();
 p2 = coords->givePointer();
 nsize = coords->giveSize();
 minDis = maxDis = dotProduct(p1,p2,nsize);


   for (i=2 ; i<=numberOfNodes ; i++) {
  coords = this->giveNode(i)->giveCoordinates();
  p2 = coords->givePointer();
  nsize = coords->giveSize();
  dis = dotProduct(p1,p2,nsize);
  if (dis>maxDis) maxDis= dis;
  else if (dis < minDis) minDis=dis;
 }
  
 return maxDis - minDis;
}

FloatArray*
Element :: ComputeMidPlaneNormal (GaussPoint*)
// valid only for plane elements (shells, plates, ....) 
// computes mid-plane normal at gaussPoint - for materials with orthotrophy
{
 _error ("Unable to compute mid-plane normal, not supported");
 return NULL;
}

int 
Element::giveIPValue (FloatArray& answer, GaussPoint* aGaussPoint, InternalStateType type, TimeStep* atTime)
{
//
  return this->giveCrossSection ()->giveIPValue (answer, aGaussPoint, type, atTime);
}

int 
Element::giveIntVarCompFullIndx (IntArray& answer, InternalStateType type)
{
//
  return this->giveCrossSection()->giveIntVarCompFullIndx (answer, type, 
                               this->giveDefaultIntegrationRulePtr()->
                               getIntegrationPoint(0)->giveMaterialMode(), this->giveMaterial());
}


InternalStateValueType
Element:: giveIPValueType (InternalStateType type)
{
//
  return this->giveCrossSection()->giveIPValueType(type, this->giveMaterial());}


int
Element:: giveIPValueSize (InternalStateType type, GaussPoint* gp) 
{
//
  return this->giveCrossSection()->giveIPValueSize (type, gp);
}


int
Element::giveSpatialDimension (void)
{
 switch(this->giveGeometryType()){
 case EGT_line_1:
 case EGT_line_2:
  return 1;
 case EGT_triangle_1:
 case EGT_triangle_2:
 case EGT_quad_1:
// case EGT_quad_2:
  return 2;
  case EGT_tetra_1:
//  case EGT_tetra_2:
 case EGT_hexa_1:
// case EGT_hexa_2:
  return 3;
 case EGT_unknown:
  break;
 }

 _error("giveSpatialDimension: failure (maybe new element type was registered)");
 return 0; //to make compiler happy
}


int
Element::giveNumberOfBoundarySides (void)
{
 switch(this->giveGeometryType()){
 case EGT_line_1:
 case EGT_line_2:
  return 2;
 case EGT_triangle_1:
 case EGT_triangle_2:
  return 3;
 case EGT_quad_1:
// case EGT_quad_2:
  return 4;
  case EGT_tetra_1:
//  case EGT_tetra_2:
  return 4;
 case EGT_hexa_1:
// case EGT_hexa_2:
  return 6;
 case EGT_unknown:
  break;
 }

 _error("giveSpatialDimension: failure (maybe new element type was registered)");
 return 0; // to make compiler happy
}


//


#ifdef __PARALLEL_MODE
int 
Element::packUnknowns (CommunicationBuffer& buff, TimeStep* stepN)
{
 int  i,j, result = 1 ;
 //GaussPoint  *gp ;
 IntegrationRule* iRule;
 
 for (i =0; i < numberOfIntegrationRules; i++) {
  iRule = integrationRulesArray[i];
  for (j=0; j < iRule->getNumberOfIntegrationPoints(); j++)
   result &= this->giveCrossSection()->packUnknowns (buff, stepN, iRule->getIntegrationPoint(j));
 }
 return result;
}
 
 
int 
Element :: unpackAndUpdateUnknowns (CommunicationBuffer& buff, TimeStep* stepN) 
{
 int  i,j, result = 1 ;
 //GaussPoint  *gp ;
 IntegrationRule* iRule;
 
 for (i =0; i < numberOfIntegrationRules; i++) {
  iRule = integrationRulesArray[i];
  for (j=0; j < iRule->getNumberOfIntegrationPoints(); j++)
   result &= this->giveCrossSection()->unpackAndUpdateUnknowns (buff, stepN, iRule->getIntegrationPoint(j));
 }
 return result;
}

int 
Element :: estimatePackSize (CommunicationBuffer& buff) 
{
 int  i,j, result = 0 ;
 IntegrationRule* iRule;
 
 for (i =0; i < numberOfIntegrationRules; i++) {
  iRule = integrationRulesArray[i];
  for (j=0; j < iRule->getNumberOfIntegrationPoints(); j++)
   result += this->giveCrossSection()->estimatePackSize (buff, iRule->getIntegrationPoint(j));
 }
 return result;
}
#endif



#ifdef __OOFEG
void
Element ::drawYourself (oofegGraphicContext& gc)
{
 OGC_PlotModeType mode = gc.giveIntVarPlotMode();

  if (mode == OGC_rawGeometry) this->drawRawGeometry(gc);
  else if (mode == OGC_deformedGeometry) 
  this->drawDeformedGeometry(gc,DisplacementVector);
 else if (mode == OGC_eigenVectorGeometry) 
  this->drawDeformedGeometry(gc,EigenVector);
  else if (mode == OGC_scalarPlot)
  this-> drawScalar (gc);
 else if (mode == OGC_elemSpecial)
  this-> drawSpecial (gc);
  else _error ("drawYourself : unsupported mode");
}


int   
Element::giveInternalStateAtNode (FloatArray& answer, InternalStateType type, InternalStateMode mode, 
                 int node, TimeStep* atTime)
{
//
  if (mode == ISM_recovered) {
   const FloatArray* nodval;
   int result = this->giveDomain()->giveSmoother()->giveNodalVector(nodval, this->giveNode(node)->giveNumber(), 
                                    this->giveRegionNumber());
   if (nodval) answer = *nodval;
   else answer.resize(0);
   return result;
  }
  else return 0;
//
}



#endif
