/* $Header: /home/cvs/bp/oofem/oofemlib/src/fei2dtrlin.C,v 1.1.4.1 2004/04/05 15:19:43 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/

#include "fei2dtrlin.h"
#include "flotarry.h"
#include "flotmtrx.h"
#include "intarray.h"
#include "node.h"
#include "mathfem.h"

void
FEI2dTrLin::evalN (FloatArray& answer, const FloatArray& lcoords, double time)
{

 answer.resize (3);
 
 answer.at(1) = lcoords.at(1);
 answer.at(2) = lcoords.at(2);
 answer.at(3) = 1. - lcoords.at(1) - lcoords.at(2);
 
 return;
}

void
FEI2dTrLin :: evaldNdx (FloatMatrix&answer, Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
 Node *node1, *node2,*node3;
 double x1,x2,x3,y1,y2,y3,area;
 answer.resize(3,2);

 node1 = d -> giveNode(nodes.at(1)) ;
 node2 = d -> giveNode(nodes.at(2)) ;
 node3 = d -> giveNode(nodes.at(3)) ;
 
 x1 = node1 -> giveCoordinate(xind) ;
 x2 = node2 -> giveCoordinate(xind) ;
 x3 = node3 -> giveCoordinate(xind) ;
 
 y1 = node1 -> giveCoordinate(yind) ;
 y2 = node2 -> giveCoordinate(yind) ;
 y3 = node3 -> giveCoordinate(yind) ;
 
 area = 0.5*(x2*y3+x1*y2+y1*x3-x2*y1-x3*y2-x1*y3) ;

 answer.at(1,1) = (y2-y3)/(2.*area);
 answer.at(1,2) = (x3-x2)/(2.*area);
 
 answer.at(2,1) = (y3-y1)/(2.*area);
 answer.at(2,2) = (x1-x3)/(2.*area);
 
 answer.at(3,1) = (y1-y2)/(2.*area);
 answer.at(3,2) = (x2-x1)/(2.*area);
}

void
FEI2dTrLin :: local2global (FloatArray& answer, Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
   double l1,l2,l3;
  answer.resize (2);
   
   l1 = lcoords.at(1);
   l2 = lcoords.at(2);
   l3 = 1.0 - l1 - l2;
  
  answer.at(1) = (l1*d->giveNode(nodes.at(1))->giveCoordinate(xind)+
          l2*d->giveNode(nodes.at(2))->giveCoordinate(xind)+
          l3*d->giveNode(nodes.at(3))->giveCoordinate(xind));
  answer.at(2) = (l1*d->giveNode(nodes.at(1))->giveCoordinate(yind)+
          l2*d->giveNode(nodes.at(2))->giveCoordinate(yind) +
          l3*d->giveNode(nodes.at(3))->giveCoordinate(yind));
}

#define POINT_TOL 1.e-3

int
FEI2dTrLin :: global2local (FloatArray& answer, Domain* d, IntArray& nodes, const FloatArray& coords, double time)
{
   Node    *node1,*node2,*node3;
   double  area, x1,x2,x3,y1,y2,y3;
  answer.resize(3);

   node1 = d -> giveNode(nodes.at(1)) ;
   node2 = d -> giveNode(nodes.at(2)) ;
   node3 = d -> giveNode(nodes.at(3)) ;

   x1 = node1 -> giveCoordinate(xind) ;
   x2 = node2 -> giveCoordinate(xind) ;
   x3 = node3 -> giveCoordinate(xind) ;

   y1 = node1 -> giveCoordinate(yind) ;
   y2 = node2 -> giveCoordinate(yind) ;
   y3 = node3 -> giveCoordinate(yind) ;

  area = 0.5*(x2*y3+x1*y2+y1*x3-x2*y1-x3*y2-x1*y3);
  

  answer.at(1) = ((x2*y3-x3*y2) + (y2-y3)*coords.at(xind) + (x3-x2)*coords.at(yind))/2./area;
  answer.at(2) = ((x3*y1-x1*y3) + (y3-y1)*coords.at(xind) + (x1-x3)*coords.at(yind))/2./area;
  answer.at(3) = ((x1*y2-x2*y1) + (y1-y2)*coords.at(xind) + (x2-x1)*coords.at(yind))/2./area;

  // check if point is inside
  int i;
  for (i=1; i<=3; i++) {
   if (answer.at(i)<(0.-POINT_TOL)) return 0;
   if (answer.at(i)>(1.+POINT_TOL)) return 0;
  }
  return 1;
}


double
FEI2dTrLin :: giveTransformationJacobian (Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
   Node    *node1,*node2,*node3;
   double  area, x1,x2,x3,y1,y2,y3;

   node1 = d -> giveNode(nodes.at(1)) ;
   node2 = d -> giveNode(nodes.at(2)) ;
   node3 = d -> giveNode(nodes.at(3)) ;

   x1 = node1 -> giveCoordinate(xind) ;
   x2 = node2 -> giveCoordinate(xind) ;
   x3 = node3 -> giveCoordinate(xind) ;

   y1 = node1 -> giveCoordinate(yind) ;
   y2 = node2 -> giveCoordinate(yind) ;
   y3 = node3 -> giveCoordinate(yind) ;

  area = 0.5*(x2*y3+x1*y2+y1*x3-x2*y1-x3*y2-x1*y3);
  return 2.0*area;
}


void 
FEI2dTrLin :: edgeEvalN (FloatArray& answer, const FloatArray& lcoords, double time)
{
 double ksi = lcoords.at(1);
 answer.resize(2);
 
 answer.at(1) = (1. - ksi) * 0.5 ;
 answer.at(2) = (1. + ksi) * 0.5 ;
}

void 
FEI2dTrLin :: edgeEvaldNdx (FloatMatrix&answer, int iedge, 
              Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
 double coeff,l,x1,x2,y1,y2;
 IntArray edgeNodes;
 this->computeEdgeMapping (edgeNodes, nodes, iedge);
 l = this->edgeComputeLength(edgeNodes, d);
 coeff = 1.0/l/l;

 x1 = d->giveNode(edgeNodes.at(1))->giveCoordinate(1);
 y1 = d->giveNode(edgeNodes.at(1))->giveCoordinate(2);
 x2 = d->giveNode(edgeNodes.at(2))->giveCoordinate(1);
 y2 = d->giveNode(edgeNodes.at(2))->giveCoordinate(2);

 answer.resize (2,2);
 answer.at(1,1) = (x1-x2)*coeff;
 answer.at(1,2) = (y1-y2)*coeff;

 answer.at(2,1) = (x2-x1)*coeff;
 answer.at(2,2) = (y2-y1)*coeff;
}

void
FEI2dTrLin :: edgeLocal2global (FloatArray& answer, int iedge, 
                Domain* d, IntArray& nodes, const FloatArray& lcoords, double time) 
{
 IntArray edgeNodes;
 FloatArray n;
 this->computeEdgeMapping (edgeNodes, nodes, iedge);
 this->edgeEvalN (n, lcoords, time);
 
 answer.resize(2);
 answer.at(1) = (n.at(1)* d->giveNode(edgeNodes.at(1))->giveCoordinate(xind) +
         n.at(2)* d->giveNode(edgeNodes.at(2))->giveCoordinate(xind));
 answer.at(2) = (n.at(1)* d->giveNode(edgeNodes.at(1))->giveCoordinate(yind) +
         n.at(2)* d->giveNode(edgeNodes.at(2))->giveCoordinate(yind));
 
}


double
FEI2dTrLin :: edgeGiveTransformationJacobian (int iedge, Domain* d, IntArray& nodes, const FloatArray& lcoords, double time) 
{
 IntArray edgeNodes;
 this->computeEdgeMapping (edgeNodes, nodes, iedge);
 return 0.5 * this->edgeComputeLength(edgeNodes, d);
}


void 
FEI2dTrLin :: computeEdgeMapping (IntArray& edgeNodes, IntArray& elemNodes, int iedge)
{
 int aNode=0, bNode=0;
 edgeNodes.resize(2);
 
  if (iedge == 1) { // edge between nodes 1 2
  aNode = elemNodes.at(1);
  bNode = elemNodes.at(2);
  } else if (iedge == 2) { // edge between nodes 2 3
  aNode = elemNodes.at(2);
  bNode = elemNodes.at(3);
  } else if (iedge == 3) { // edge between nodes 2 3
  aNode = elemNodes.at(3);
  bNode = elemNodes.at(1);
  } else {
  OOFEM_ERROR2 ("FEI2dTrLin :: computeEdgeMapping: wrong egde number (%d)", iedge);
  }
 
 edgeNodes.at(1) = aNode;
 edgeNodes.at(2) = bNode;
}

double
FEI2dTrLin :: edgeComputeLength (IntArray& edgeNodes, Domain* d)
{
 double dx,dy;
  Node   *nodeA,*nodeB ;

  nodeA   = d->giveNode(edgeNodes.at(1)) ;
  nodeB   = d->giveNode(edgeNodes.at(2)) ;
 
  dx      = nodeB->giveCoordinate(xind) - nodeA->giveCoordinate(xind) ;
  dy      = nodeB->giveCoordinate(yind) - nodeA->giveCoordinate(yind) ;
  return (sqrt(dx*dx + dy*dy));
}
