/* $Header: /home/cvs/bp/oofem/oofemlib/src/fei3dtrlin.C,v 1.1.4.1 2004/04/05 15:19:43 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/

#include "fei3dtrlin.h"
#include "flotarry.h"
#include "flotmtrx.h"
#include "intarray.h"
#include "node.h"
#include "mathfem.h"

void
FEI3dTrLin::evalN (FloatArray& answer, const FloatArray& lcoords, double time)
{

 answer.resize (4);
 
 answer.at(1) = lcoords.at(1);
 answer.at(2) = lcoords.at(2);
 answer.at(3) = lcoords.at(3);
 answer.at(4) = 1. - lcoords.at(1) - lcoords.at(2) - lcoords.at(3);
 
 return;
}

void
FEI3dTrLin :: evaldNdx (FloatMatrix&answer, Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
 Node *node1, *node2,*node3, *node4;
 double x1,x2,x3,x4, y1,y2,y3,y4, z1,z2,z3,z4, vol;
 answer.resize(4,3);

 node1 = d -> giveNode(nodes.at(1)) ;
 node2 = d -> giveNode(nodes.at(2)) ;
 node3 = d -> giveNode(nodes.at(3)) ;
 node4 = d -> giveNode(nodes.at(4)) ;
 
 x1 = node1 -> giveCoordinate(1) ;
 x2 = node2 -> giveCoordinate(1) ;
 x3 = node3 -> giveCoordinate(1) ;
 x4 = node4 -> giveCoordinate(1) ;
 
 y1 = node1 -> giveCoordinate(2) ;
 y2 = node2 -> giveCoordinate(2) ;
 y3 = node3 -> giveCoordinate(2) ;
 y4 = node4 -> giveCoordinate(2) ;

 z1 = node1 -> giveCoordinate(3) ;
 z2 = node2 -> giveCoordinate(3) ;
 z3 = node3 -> giveCoordinate(3) ;
 z4 = node4 -> giveCoordinate(3) ;
 
 vol = ((x4-x1)*(y2-y1)*(z3-z1) - (x4-x1)*(y3-y1)*(z2-z1) +
     (x3-x1)*(y4-y1)*(z2-z1) - (x2-x1)*(y4-y1)*(z3-z1) +
     (x2-x1)*(y3-y1)*(z4-z1) - (x3-x1)*(y2-y1)*(z4-z1))/6.;
 
 if (vol <= 0.0) {
  OOFEM_ERROR ("FEI3dTrLin :: evaldNdx: negative volume");
 }

 answer.at(1,1) = -((y3-y2)*(z4-z2)-(y4-y2)*(z3-z2));
 answer.at(2,1) = (y4-y3)*(z1-z3)-(y1-y3)*(z4-z3);
 answer.at(3,1) = -((y1-y4)*(z2-z4)-(y2-y4)*(z1-z4));
 answer.at(4,1) = (y2-y1)*(z3-z1)-(y3-y1)*(z2-z1);
 
 answer.at(1,2) = -((x4-x2)*(z3-z2)-(x3-x2)*(z4-z2));
 answer.at(2,2) = (x1-x3)*(z4-z3)-(x4-x3)*(z1-z3);
 answer.at(3,2) = -((x2-x4)*(z1-z4)-(x1-x4)*(z2-z4));
 answer.at(4,2) = (x3-x1)*(z2-z1)-(x2-x1)*(z3-z1);
 
 answer.at(1,3) = -((x3-x2)*(y4-y2)-(x4-x2)*(y3-y2));
 answer.at(2,3) = (x4-x3)*(y1-y3)-(x1-x3)*(y4-y3);
 answer.at(3,3) = -((x1-x4)*(y2-y4)-(x2-x4)*(y1-y4));
 answer.at(4,3) = (x2-x1)*(y3-y1)-(x3-x1)*(y2-y1);
 
 answer.times(1./(6.*vol));
}

void
FEI3dTrLin :: local2global (FloatArray& answer, Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
 int i;
 FloatArray l(4);
 FloatArray* c;
 answer.resize (3); 
 answer.zero();
 
 l.at(1) = lcoords.at(1);
 l.at(2) = lcoords.at(2);
 l.at(3) = lcoords.at(3);
 l.at(4) = 1.0 - l.at(1) - l.at(2) - l.at(3);
 
 for (i=1; i<=4; i++) { 
  c = d->giveNode(nodes.at(i))->giveCoordinates();
  answer.at(1) += l.at(i)*c->at(1);
  answer.at(2) += l.at(i)*c->at(2);
  answer.at(3) += l.at(i)*c->at(3);
 }

}

#define POINT_TOL 1.e-3

int
FEI3dTrLin :: global2local (FloatArray& answer, Domain* d, IntArray& nodes, const FloatArray& coords, double time)
{
   Node    *node1,*node2,*node3, *node4;
  double       x1,x2,x3,x4,y1,y2,y3,y4,z1,z2,z3,z4,xp,yp,zp, volume;
  answer.resize(4);

   node1 = d -> giveNode(nodes.at(1)) ;
   node2 = d -> giveNode(nodes.at(2)) ;
   node3 = d -> giveNode(nodes.at(3)) ;
  node4 = d -> giveNode(nodes.at(4)) ;

  x1 = node1 -> giveCoordinate(1) ;
  x2 = node2 -> giveCoordinate(1) ;
  x3 = node3 -> giveCoordinate(1) ;
  x4 = node4 -> giveCoordinate(1) ;
  
  y1 = node1 -> giveCoordinate(2) ;
  y2 = node2 -> giveCoordinate(2) ;
  y3 = node3 -> giveCoordinate(2) ;
  y4 = node4 -> giveCoordinate(2) ;
  
  z1 = node1 -> giveCoordinate(3) ;
  z2 = node2 -> giveCoordinate(3) ;
  z3 = node3 -> giveCoordinate(3) ;
  z4 = node4 -> giveCoordinate(3) ;
  
  xp= coords.at(1);
  yp= coords.at(2);
  zp= coords.at(3);
  
  volume = ((x4-x1)*(y2-y1)*(z3-z1) - (x4-x1)*(y3-y1)*(z2-z1) +
     (x3-x1)*(y4-y1)*(z2-z1) - (x2-x1)*(y4-y1)*(z3-z1) +
     (x2-x1)*(y3-y1)*(z4-z1) - (x3-x1)*(y2-y1)*(z4-z1))/6.;

  answer.resize(4);
  
  answer.at(1) = ((x3-x2)*(yp-y2)*(z4-z2) - (xp-x2)*(y3-y2)*(z4-z2) +
          (x4-x2)*(y3-y2)*(zp-z2) - (x4-x2)*(yp-y2)*(z3-z2) +
          (xp-x2)*(y4-y2)*(z3-z2) - (x3-x2)*(y4-y2)*(zp-z2))/6./volume;
  
  answer.at(2) = ((x4-x1)*(yp-y1)*(z3-z1) - (xp-x1)*(y4-y1)*(z3-z1) +
          (x3-x1)*(y4-y1)*(zp-z1) - (x3-x1)*(yp-y1)*(z4-z1) +
          (xp-x1)*(y3-y1)*(z4-z1) - (x4-x1)*(y3-y1)*(zp-z1))/6./volume;
  
  answer.at(3) = ((x2-x1)*(yp-y1)*(z4-z1) - (xp-x1)*(y2-y1)*(z4-z1) +
         (x4-x1)*(y2-y1)*(zp-z1) - (x4-x1)*(yp-y1)*(z2-z1) +
          (xp-x1)*(y4-y1)*(z2-z1) - (x2-x1)*(y4-y1)*(zp-z1))/6./volume;
  
  answer.at(4) = 1.0-answer.at(1)-answer.at(2)-answer.at(3);
  
  // test if inside
  for (int i=1; i<=4; i++) {
   if (answer.at(i)<(0.-POINT_TOL)) return 0;
   if (answer.at(i)>(1.+POINT_TOL)) return 0;
  }
  return 1;
 }


double
FEI3dTrLin :: giveTransformationJacobian (Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
 Node         *node1,*node2,*node3,*node4;
 double       volume, x1,x2,x3,x4,y1,y2,y3,y4,z1,z2,z3,z4;

 node1 = d -> giveNode(nodes.at(1)) ;
 node2 = d -> giveNode(nodes.at(2)) ;
 node3 = d -> giveNode(nodes.at(3)) ;
 node4 = d -> giveNode(nodes.at(4)) ;
 
 x1 = node1 -> giveCoordinate(1) ;
 x2 = node2 -> giveCoordinate(1) ;
 x3 = node3 -> giveCoordinate(1) ;
 x4 = node4 -> giveCoordinate(1) ;
 
 y1 = node1 -> giveCoordinate(2) ;
 y2 = node2 -> giveCoordinate(2) ;
 y3 = node3 -> giveCoordinate(2) ;
 y4 = node4 -> giveCoordinate(2) ;
 
 z1 = node1 -> giveCoordinate(3) ;
 z2 = node2 -> giveCoordinate(3) ;
 z3 = node3 -> giveCoordinate(3) ;
 z4 = node4 -> giveCoordinate(3) ;
 
 volume = ((x4-x1)*(y2-y1)*(z3-z1) - (x4-x1)*(y3-y1)*(z2-z1) +
      (x3-x1)*(y4-y1)*(z2-z1) - (x2-x1)*(y4-y1)*(z3-z1) +
      (x2-x1)*(y3-y1)*(z4-z1) - (x3-x1)*(y2-y1)*(z4-z1))/6.;
 
 if (volume <= 0.0) {
  OOFEM_ERROR ("FEI3dTrLin :: giveTransformationJacobian: negative volume encountered");
 }
 return volume;
}


void 
FEI3dTrLin :: edgeEvalN (FloatArray& answer, const FloatArray& lcoords, double time)
{
 double ksi = lcoords.at(1);
 answer.resize(2);
 
 answer.at(1) = (1. - ksi) * 0.5 ;
 answer.at(2) = (1. + ksi) * 0.5 ;
}

void 
FEI3dTrLin :: edgeEvaldNdx (FloatMatrix&answer, int iedge, 
              Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
 double coeff,l,x1,x2,y1,y2,z1,z2;
 IntArray edgeNodes;
 this->computeEdgeMapping (edgeNodes, nodes, iedge);
 l = this->edgeComputeLength(edgeNodes, d);
 coeff = 1.0/l/l;

 x1 = d->giveNode(edgeNodes.at(1))->giveCoordinate(1);
 y1 = d->giveNode(edgeNodes.at(1))->giveCoordinate(2);
 z1 = d->giveNode(edgeNodes.at(1))->giveCoordinate(3);
 x2 = d->giveNode(edgeNodes.at(2))->giveCoordinate(1);
 y2 = d->giveNode(edgeNodes.at(2))->giveCoordinate(2);
 z2 = d->giveNode(edgeNodes.at(2))->giveCoordinate(3);

 answer.resize (2,3);
 answer.at(1,1) = (x1-x2)*coeff;
 answer.at(1,2) = (y1-y2)*coeff;
 answer.at(1,3) = (z1-z2)*coeff;

 answer.at(2,1) = (x2-x1)*coeff;
 answer.at(2,2) = (y2-y1)*coeff;
 answer.at(2,3) = (z2-z1)*coeff;
}

void
FEI3dTrLin :: edgeLocal2global (FloatArray& answer, int iedge, 
                Domain* d, IntArray& nodes, const FloatArray& lcoords, double time) 
{
 IntArray edgeNodes;
 FloatArray n;
 this->computeEdgeMapping (edgeNodes, nodes, iedge);
 this->edgeEvalN (n, lcoords, time);
 
 answer.resize(3); 
 answer.at(1) = (n.at(1)* d->giveNode(edgeNodes.at(1))->giveCoordinate(1) +
         n.at(2)* d->giveNode(edgeNodes.at(2))->giveCoordinate(1));
 answer.at(2) = (n.at(1)* d->giveNode(edgeNodes.at(1))->giveCoordinate(2) +
         n.at(2)* d->giveNode(edgeNodes.at(2))->giveCoordinate(2));
 answer.at(3) = (n.at(1)* d->giveNode(edgeNodes.at(1))->giveCoordinate(3) +
         n.at(2)* d->giveNode(edgeNodes.at(2))->giveCoordinate(3));
  
}


double
FEI3dTrLin :: edgeGiveTransformationJacobian (int iedge, Domain* d, IntArray& nodes, const FloatArray& lcoords, double time) 
{
 IntArray edgeNodes;
 this->computeEdgeMapping (edgeNodes, nodes, iedge);
 return 0.5 * this->edgeComputeLength(edgeNodes, d);
}


void 
FEI3dTrLin :: computeEdgeMapping (IntArray& edgeNodes, IntArray& elemNodes, int iedge)
{
 int aNode=0, bNode=0;
 edgeNodes.resize(2);
 
  if (iedge == 1) { // edge between nodes 1 2
  aNode = 1;
  bNode = 2;
  } else if (iedge == 2) { // edge between nodes 2 3
  aNode = 2;
  bNode = 3;
  } else if (iedge == 3) { // edge between nodes 3 1
  aNode = 3;
  bNode = 1;
 } else if (iedge == 4) { // edge between nodes 1 4
  aNode = 1;
  bNode = 4;
 } else if (iedge == 5) { // edge between nodes 2 4
  aNode = 2;
  bNode = 4;
 } else if (iedge == 6) { // edge between nodes 3 4
  aNode = 3;
  bNode = 4;
  } else {
  OOFEM_ERROR2 ("FEI3dTrLin :: computeEdgeMapping: wrong egde number (%d)", iedge);
  }
 
 edgeNodes.at(1) = elemNodes.at(aNode);
 edgeNodes.at(2) = elemNodes.at(bNode);
}

double
FEI3dTrLin :: edgeComputeLength (IntArray& edgeNodes, Domain* d)
{
 double dx,dy,dz;
  Node   *nodeA,*nodeB ;

  nodeA   = d->giveNode(edgeNodes.at(1)) ;
  nodeB   = d->giveNode(edgeNodes.at(2)) ;
 
  dx      = nodeB->giveCoordinate(1) - nodeA->giveCoordinate(1) ;
  dy      = nodeB->giveCoordinate(2) - nodeA->giveCoordinate(2) ;
  dz      = nodeB->giveCoordinate(3) - nodeA->giveCoordinate(3) ;
  return (sqrt(dx*dx + dy*dy + dz*dz));
}

void
FEI3dTrLin :: surfaceEvalN (FloatArray& answer, const FloatArray& lcoords, double time)
{
 answer.resize (3);
 
 answer.at(1) = lcoords.at(1);
 answer.at(2) = lcoords.at(2);
 answer.at(3) = 1. - lcoords.at(1) - lcoords.at(2);
 
 return;
}

void 
FEI3dTrLin :: surfaceLocal2global (FloatArray& answer, int iedge,
                  Domain* d, IntArray& nodes, const FloatArray& lcoords, double time)
{
   double l1,l2,l3;
  answer.resize (3);
   
   l1 = lcoords.at(1);
   l2 = lcoords.at(2);
   l3 = 1.0 - l1 - l2;
  
  answer.at(1) = (l1*d->giveNode(nodes.at(1))->giveCoordinate(1)+
          l2*d->giveNode(nodes.at(2))->giveCoordinate(1)+
          l3*d->giveNode(nodes.at(3))->giveCoordinate(1));
  answer.at(2) = (l1*d->giveNode(nodes.at(1))->giveCoordinate(2)+
          l2*d->giveNode(nodes.at(2))->giveCoordinate(2) +
          l3*d->giveNode(nodes.at(3))->giveCoordinate(2));
  answer.at(3) = (l1*d->giveNode(nodes.at(1))->giveCoordinate(3)+
          l2*d->giveNode(nodes.at(2))->giveCoordinate(3) +
          l3*d->giveNode(nodes.at(3))->giveCoordinate(3));
}

double 
FEI3dTrLin :: surfaceGiveTransformationJacobian (int isurf, Domain* d, IntArray& nodes, const FloatArray& lcoords,
                         double time)
{
 Node *n1,*n2,*n3;
 IntArray snodes (3);
 this->computeSurfaceMapping (snodes, nodes, isurf);

 n1 = d->giveNode(snodes.at(1));
 n2 = d->giveNode(snodes.at(2));
 n3 = d->giveNode(snodes.at(3));

 FloatArray a(3),b(3),c(3);
 for (int i=1; i<=3; i++) {
  a.at(i) = n2->giveCoordinate(i)-n1->giveCoordinate(i);
  b.at(i) = n3->giveCoordinate(i)-n1->giveCoordinate(i);
 }
 
 c.beVectorProductOf (a,b);
 return 0.5 * sqrt(dotProduct (c,c,3));
}

void 
FEI3dTrLin :: computeSurfaceMapping (IntArray& surfNodes, IntArray& elemNodes, int isurf)
{
 int aNode=0, bNode=0, cNode=0;
 surfNodes.resize(3);
 
  if (isurf == 1) { // surface 1 - nodes 1 3 2
  aNode = 1;
  bNode = 3;
  cNode = 2;
  } else if (isurf == 2) { // surface 2 - nodes 1 2 4
  aNode = 1;
  bNode = 2;
  cNode = 4;
  } else if (isurf == 3) { // surface 3  - nodes 2 3 4
  aNode = 2;
  bNode = 3;
  cNode = 4;
 } else if (isurf == 4) { // surface 4 - nodes 1 4 3
  aNode = 1;
  bNode = 4;
  cNode = 3;
 } else {
  OOFEM_ERROR2 ("FEI3dTrLin :: computeSurfaceMapping: wrong surface number (%d)", isurf);
 }
 
 surfNodes.at(1) = elemNodes.at(aNode);
 surfNodes.at(2) = elemNodes.at(bNode);
 surfNodes.at(3) = elemNodes.at(cNode);
}



