/* $Header: /home/cvs/bp/oofem/oofemlib/src/inputrecord.h,v 1.4 2003/05/19 13:03:57 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/


//
// Class stringReader
//

#ifndef inputrecord_h

#include "cltypes.h"
#ifndef __MAKEDEPEND
#include <stdio.h>
#include <string.h>
#endif
#include "intarray.h"
#include "flotarry.h"
#include "dictionr.h"
#include "dynalist.h"
#include "range.h"

#define IR_MAX_ERROR_LENGTH 100

/** Define this directive to use tokenizer to parse records.
  This also enables to perform additional check for input records, since
  unread fields can be detected
*/
#define IR_USE_TOKENIZER

#ifdef IR_USE_TOKENIZER 
#include "tokenizer.h"
#endif

/**
 Type defining the return values of InputRecord reading operations.
 IRRT_OK the corresponding value to given keyword was successfully read.
         the answer parameter contains the value.
 IRRT_NOTFOUND the keyword is not found; the answer is not modified
 IRRT_BAD_FORMAT the keyword was found but the record is not correctly formated.
*/
enum IRResultType {IRRT_OK = 0, IRRT_NOTFOUND, IRRT_BAD_FORMAT};

/**
 Macro simplifying the erorr reporting.
*/
#define IR_IOERR(__class, __proc, __keyword, __ir, __result) \
__ir->report_error (__class, __proc, __keyword, __result, __FILE__, __LINE__);
 
/**
 Macro facilitating the use of input record reading methods. 
 uses the given input record (__ir parameter) and reads the compulsory
 field identified by __kwd and stores the  result into __value parameter. 
 Includes also the erorr reporting. 
*/
#define IR_GIVE_FIELD(__ir, __value, __kwd) __keyword = __kwd; result = __ir->giveField(__value, __keyword); \
if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, __keyword, __ir, result);
 
/**
 Macro facilitating the use of input record reading methods. 
 uses the given input record (__ir parameter) and reads the optional
 field identified by __kwd and stores the  result into __value parameter. 
 Includes also the erorr reporting. 
*/
#define IR_GIVE_OPTIONAL_FIELD(__ir, __value, __kwd) __keyword = __kwd; result = __ir->giveOptionalField(__value, __keyword); \
if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, __keyword, __ir, result);

/**
 Macro facilitating the use of input record reading methods. 
 uses the given input record (__ir parameter) and reads the compulsory
 field identified by __kwd and stores the  result into __value parameter. 
 Includes also the erorr reporting. 
*/
#define IR_GIVE_FIELD2(__ir, __value, __kwd, __opt) __keyword = __kwd; result = __ir->giveField(__value, __opt, __keyword); \
if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, __keyword, __ir, result);
 
/**
 Macro facilitating the use of input record reading methods. 
 uses the given input record (__ir parameter) and reads the optional
 field identified by __kwd and stores the  result into __value parameter. 
 Includes also the erorr reporting. 
*/
#define IR_GIVE_OPTIONAL_FIELD2(__ir, __value, __kwd, __opt) __keyword = __kwd; \
result = __ir->giveOptionalField(__value, __opt, __keyword); \
if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, __keyword, __ir, result);
/**
 Macro facilitating the use of input record reading methods. 
 uses the given input record (__ir parameter) and reads the compulsory record keyword (__kwd) 
 and its number (__value param). Includes also the erorr reporting. 
*/
#define IR_GIVE_RECORD_KEYWORD_FIELD(__ir, __kwd, __value, __opt) \
result = __ir->giveRecordKeywordField(__kwd, __value, __opt); \
if (result != IRRT_OK) IR_IOERR (giveClassName(), __proc, "RECORD_KEYWORD", __ir, result);
/**
 Class representing the general Input Record. The input record consis of several fields.
 Provides several requesting functions for reading field values. The derived classes of 
 Input record can represent database records or text file records, allowing the transparent 
 input operations. Currently this class inplements the text file record functionality,
 as the text file input is the only supported. 
*/
class InputRecord
{
protected:
 /** Defines Input Reader Result Type. Values of this type are the return parameters 
  of several read-like services of this class. Possible values are:
  IRRT_OK .. the corresponding value to given keyword was successfully read.
            the answer parameter contains the value.
  IRRT_COMPULSORY_KEYWORD_NOT_FOUND ... the keyword is not found; the answer is not modified
  IRRT_BAD_FORMAT ... the keyword was found but the record is not correctly formated.
  */
 
 char record[OOFEM_MAX_LINE_LENGTH];
#ifdef IR_USE_TOKENIZER 
 Tokenizer tokenizer;
 bool readFlag[MAX_TOKENS];
#endif

public:
 /// Constructor. Creates an empty input record.
 InputRecord ();
 /// Constructor. Creates the input record corresponding to given string
 InputRecord (char* source) ;
 /// Copy constructor
 InputRecord (const InputRecord&); 
 /// Destructor
 ~InputRecord () {}
 // Assingnment operator
 InputRecord&  operator=  (const InputRecord&);

 /**@name Compulsory field extraction methods
  Reads the field value identified by keyword
  @param answer contains result
  @param idString field keyword
  @return IRResultType 
  */
  //@{
 /// Reads the record keyword (entity record) and its corresponding number
 IRResultType giveRecordKeywordField(char* answer, int& value, int maxchar);
 /// Reads the integer field value
 IRResultType giveField (int& answer, const char* idString);
 /// Reads the double field value
 IRResultType giveField (double& answer, const char* idString);
 /// Reads the char* field value
 IRResultType giveField (char* answer, int maxchar, const char* idString);
 /// Reads the FloatArray field value
 IRResultType giveField (FloatArray& answer, const char* idString);
 /// Reads the IntArray field value
 IRResultType giveField (IntArray& answer, const char* idString);
 /// Reads the Dictionary field value
 IRResultType giveField (Dictionary& answer, const char* idString);
 /// Reads the dynaList<Range> field value
 IRResultType giveField (dynaList<Range> &answer, const char* idString);
 //@}

 /**@name Optional field extraction methods
  Reads the field value identified by keyword
  @param answer contains result
  @param idString field keyword
  @return IRResultType 
  */
 //@{
 /// Reads the integer field value
 IRResultType giveOptionalField (int& answer, const char* idString);
 /// Reads the double field value
 IRResultType giveOptionalField (double& answer, const char* idString);
 /// Reads the char* field value
 IRResultType giveOptionalField (char* answer, int maxchar, const char* idString);
 /// Reads the FloatArray field value
 IRResultType giveOptionalField (FloatArray& answer, const char* idString);
 /// Reads the IntArray field value
 IRResultType giveOptionalField (IntArray& answer, const char* idString);
 /// Reads the Dictionary field value
 IRResultType giveOptionalField (Dictionary& answer, const char* idString);
 /// Reads the dynaList<Range> field value
 IRResultType giveOptionalField (dynaList<Range> &answer, const char* idString);
 //@}

 /// Returns true if record contains field identified by idString keyword
 bool         hasField  (const char* idString);
 
 /// Sets the record string
 void setRecordString (char*);
 /// Returns record string
 char* giveRecordAsString () {return this->record;}
 /// Returns error string corresponding to given value of IRResultType type
 char* strerror (IRResultType); 

 /// Prints the error message 
 void report_error (const char* _class, const char* proc, const char* kwd, IRResultType result, const char* file, int line);

 /** terminates the current record session and if flag is true warnin is printed for unscanned tokens */
 void finish (bool wrn = true);

protected:

#ifdef IR_USE_TOKENIZER 
  int giveKeywordIndx (const char* kwd);
  int scanInteger (const char* source, int& value);
  int scanDouble  (const char* source, double& value);
  void setReadFlag (int itok) {readFlag[itok-1]=true;}
 
#endif

#ifndef IR_USE_TOKENIZER 

  char* getPosAfter (char*, const char *) ;
  char* scanInteger (char* source, int* value);
  char* scanDouble  (char* source, double* value);
  char* skipNextWord (char* src) ;

  char*  readSimpleString (char* source, char* simpleString, int maxchar, char** remain); 
  char*  readKeyAndVal    (char* source, char* key, int* val, int maxchar, char** remain);
  char*  readKeyAndVal    (char* source, char* key, double* val, int maxchar, char** remain);
#endif
 /**
  Reads single range record from input record represented by *helpSource  string.
  @param helpSource pointer to current string possition, on return helpSource points
  to next charcter after reading range record.
  @param li starting range index
  @param hi end range index
  @return on success nonzero valur returned
  */
 int    readRange (const char** helpSource, int& li, int& hi);

};

#define inputrecord_h
#endif
