/* $Header: /home/cvs/bp/oofem/oofemlib/src/masterdof.C,v 1.14.4.1 2004/04/05 15:19:43 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/


//   file MASTERDOF.CC

#include "masterdof.h"
#include "dofmanager.h"
#include "domain.h"
#include "timestep.h"
#include "boundary.h"
#include "initial.h"

#include "flotarry.h"
#include "dictionr.h"

#include "debug.h"
#include "cltypes.h"
#ifndef __MAKEDEPEND
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#endif


MasterDof :: MasterDof (int i, DofManager* aNode, int nbc, int nic, DofID id) : Dof (i, aNode, id)
   // Constructor. Creates a new d.o.f., with number i, belonging
   // to aNode with bc=nbc, ic=nic
{
   equationNumber = 0 ;                         // means "uninitialized"
   bc             = nbc ;
   ic             = nic ;
 unknowns       = new Dictionary ();
/*   unknowns       = new Dictionary() ;           // unknown size ?
   pastUnknowns   = NULL ; */
}


BoundaryCondition*  MasterDof :: giveBc () 
   // Returns the boundary condition the receiver is subjected to.
{
#  ifdef DEBUG
      if (bc == -1) {
    _error (" does not know yet if has bc or not \n") ;
    exit(0) ;}
#  endif

   return  (BoundaryCondition*) (dofManager -> giveDomain() -> giveLoad(bc)) ;
}


int  MasterDof :: giveEquationNumber () 
   // Returns the number of the equation in the governing system of equations that corres-
   // ponds to the receiver. The equationNumber is <0 if the receiver is
   // subjected to a boundary condition and then the equationNumber is a Prescribed equation number
  // or equationNumber >0.
 
{

  // EngngModel* model;
  // model = (EngngModel*) (dofManager->giveDomain()->giveEngngModel());

  if (!equationNumber) 
    _error ("giveEquationNumber : Dof has undefined equationNumber");
 //equationNumber = this->askNewEquationNumber();
  return (equationNumber>0)?equationNumber:0 ;
}

int  MasterDof :: givePrescribedEquationNumber () 
   // Returns the number of the equation in the governing system of equations that corres-
   // ponds to the receiver. The equationNumber is <0 if the receiver is
   // subjected to a boundary condition and then the equationNumber is a Prescribed equation number
  // or equationNumber >0.
{

  // EngngModel* model;
  // model = (EngngModel*) (dofManager->giveDomain()->giveEngngModel());

  if (!equationNumber) 
    _error ("giveEquationNumber: Dof has undefined equationNumber");
 //equationNumber = this->askNewEquationNumber();
  return (equationNumber<0)?-1*equationNumber:0 ;
}

int  MasterDof :: askNewEquationNumber (TimeStep* tStep)
   // Returns the newly obtained number of the equation in the governing system
  // of equations that corres-
   // ponds to the receiver. The equation number is 0 if the receiver is
   // subjected to a boundary condition, else it is n+1, where n is the
   // equation number of the most recently numbered degree of freedom.
{

  EngngModel* model;
  model = (EngngModel*) (dofManager->giveDomain()->giveEngngModel());

 if (this->hasBc(tStep)) equationNumber = 
  -1*model -> giveNewPrescribedEquationNumber (dofManager->giveDomain()->giveNumber());
 else equationNumber = 
  model -> giveNewEquationNumber (dofManager->giveDomain()->giveNumber());

 return equationNumber ;
}


InitialCondition*  MasterDof :: giveIc () 
   // Returns the initial condition on the receiver. Not used.
{
#  ifdef DEBUG
      if (ic == -1) {
    _error ("giveIc:  does not know yet if has InitCond or not \n") ;
    exit(0) ;}
#  endif

   return  (InitialCondition*) (dofManager -> giveDomain() -> giveLoad(ic)) ;
}


double  MasterDof :: giveUnknown (UnknownType type, ValueModeType mode, TimeStep* stepN) 
   // The key method of class Dof. Returns the value of the unknown 'u'
   // (e.g., the displacement) of the receiver, at stepN. This value may,
   // or may not, be already available. It may depend on a boundary (if it
   // is not a predicted unknown) or initial condition. stepN is not the
   // current time step n, it is assumed to be the previous one (n-1).
{
/*   double value ;

   if (stepN -> isTheCurrentTimeStep()) {
      if (unknowns -> includes(u))                       // already known
  value = unknowns -> at(u) ;
      else {
  if (stepN->giveNumber()==0) {                   // step 0
     if (this->hasIcOn(u))                        //   init. cond.
        value = this -> giveIc() -> give(u) ;
     else                                         //   no init. cond.
        value = 0. ;}
  else if (this->hasBc() && islower(u))           // bound . cond.
     value = this -> giveBc() -> give(u,stepN) ;
  else                                            // compute it !
     value = this -> computeUnknown(u,stepN) ;
  unknowns -> add(u,value) ;}}

   else
      value = this -> givePastUnknown(u,stepN) ;         // the previous step

   return value ;   */
 double value;

#ifdef DEBUG
 //if (type != this->giveUnknownType ())
 // _error ("giveUnknown: Noncompatible Request");  
#endif 

 if (dofManager -> giveDomain() -> giveEngngModel() -> requiresUnknowsDictionaryUpdate()) {
  // if this feature is active, engng model must ensure 
  // valid data in unknowns dictionary
  if (unknowns -> includes(mode)) return unknowns->at(mode);
  else _error2 ("giveUnknown:  Dof unknowns dictionary does not contain unknown of type %d", mode);
 } else {
  // first try if IC apply
  if (stepN->giveNumber() == dofManager->giveDomain()->giveEngngModel()->giveNumberOfTimeStepWhenIcApply())
   { // step when Ic apply
    if (this->hasIcOn (mode))
     value = this -> giveIc() -> give(mode) ;
    else
     value = 0.;
    
    return value;
   }
  // then ask bor BC
  if (this->hasBc(stepN))           // bound . cond.
   {
    //value = this -> giveBcValue(giveUnknownType(),mode,stepN) ;
    value = this -> giveBcValue(mode,stepN) ;
    return value ;
   } 
  // try ask emodel for unknown
  return (dofManager -> giveDomain() -> giveEngngModel() -> 
      giveUnknownComponent (type,mode,stepN,dofManager -> giveDomain(), this)) ;
 }
 return 0.0;
}

double  MasterDof :: giveUnknown (PrimaryField& field, ValueModeType mode, TimeStep* stepN) 
   // The key method of class Dof. Returns the value of the unknown field
   // (e.g., the displacement) associated to the receiver, at stepN. 
{
 double value;
 
#ifdef DEBUG
#endif 
 // first try if IC apply
 if (stepN->giveNumber() == dofManager->giveDomain()->giveEngngModel()->giveNumberOfTimeStepWhenIcApply())
  { // step when Ic apply
   if (this->hasIcOn (mode))
    value = this -> giveIc() -> give(mode) ;
   else
    value = 0.;
   
   return value;
  }
 // then ask bor BC
 if (this->hasBc(stepN))           // bound . cond.
  {
   //value = this -> giveBcValue(giveUnknownType(),mode,stepN) ;
   value = this -> giveBcValue(mode, stepN) ;
   return value ;
  } 
 // try ask field for unknown
 return (field.giveUnknownValue (this, mode, stepN)) ;
}


int  MasterDof :: hasBc (TimeStep* tStep) 
   // Returns True if the receiver is subjected to a boundary condition, else
   // returns False. If necessary, reads the answer in the data file.
{
   if (bc == -1) {
   _error ("hasBc:  does not know yet if has InitCond or not \n") ;
   exit(0) ;}

   if (bc) {
   return this -> giveBc() -> isImposed (tStep) ;
  } else return 0;
}


int  MasterDof :: hasIc ()
   // Returns True if the receiver is subjected to an initial condition,
   // else returns False.
{


   if (ic == -1) {
   _error ("hasIc:  does not know yet if has InitCond or not \n") ;
   exit(0) ;}

   return ic ;
}


int  MasterDof :: hasIcOn (ValueModeType u) 
   // Returns True if the unknown 'u' (e.g., the displacement 'd') of the
   // receiver is subjected to an initial condition, else returns False.
{
   if (this->hasIc())
      return this->giveIc()->hasConditionOn(u) ;
   else
      return FALSE ;
}

int MasterDof::giveBcIdValue ()
{
 return this->bc;
}


void  MasterDof :: updateYourself (TimeStep* tStep)
   // Updates the receiver at end of step.
{
  Dof::updateYourself (tStep);
/*   delete pastUnknowns ;
   pastUnknowns = unknowns ;
   unknowns     = new Dictionary() ; */
}

void MasterDof :: updateUnknownsDictionary (TimeStep* tStep, UnknownType type, 
                      ValueModeType mode, double dofValue)
{
 // Updates the receiver's unknown dictionary at end of step.
 // to value dofValue.
#ifdef DEBUG
// if (type != this->giveUnknownType ())
//  _error ("giveUnknown: Noncompatible Request");  
#endif 

 unknowns -> at(mode) = dofValue;
}


void  MasterDof :: printYourself ()
 // Prints the receiver on screen.
{
 printf ("dof %d  of %s %d :\n",number,dofManager->giveClassName(), dofManager->giveNumber()) ;
 printf ("equation %d    bc %d \n",equationNumber,bc) ;

 // printOutputAt (node->giveDomain()->giveEngngModel()->giveCurrentStep());
}

  
contextIOResultType MasterDof :: saveContext (FILE* stream, void *obj)
//
// saves full node context (saves state variables, that completely describe
// current state)
//
{
  contextIOResultType iores ;
  if (stream == NULL) _error ("saveContex : can't write into NULL stream");
  
  if ((iores = Dof::saveContext (stream, obj)) != CIO_OK) THROW_CIOERR(iores);
 // store equation number of receiver
 if (fwrite (&equationNumber,sizeof(int),1,stream) != 1) THROW_CIOERR(CIO_IOERR);
  if (dofManager -> giveDomain() -> giveEngngModel() -> requiresUnknowsDictionaryUpdate()) 
  if ((iores = unknowns->saveContext (stream,obj)) != CIO_OK) THROW_CIOERR(iores);

  return CIO_OK;
}


contextIOResultType MasterDof :: restoreContext (FILE* stream, void *obj)
//
// restores full node context (saves state variables, that completely describe
// current state)
//
{
  contextIOResultType iores ;
  if (stream == NULL) _error ("restoreContex : can't write into NULL stream");

  if ((iores = Dof::restoreContext (stream, obj)) != CIO_OK) THROW_CIOERR(iores);
 // read equation number of receiver
 if (fread (&equationNumber,sizeof(int),1,stream) != 1) THROW_CIOERR(CIO_IOERR);
  if (dofManager -> giveDomain() -> giveEngngModel() -> requiresUnknowsDictionaryUpdate()) 
  if ((iores = unknowns->restoreContext (stream,obj)) != CIO_OK) THROW_CIOERR(iores);
  
  return CIO_OK;
}


#ifdef __PARALLEL_MODE
int 
MasterDof :: packUnknowns (CommunicationBuffer& buff, UnknownType type, ValueModeType mode, TimeStep* stepN)
{
 return buff.packDouble (this->giveUnknown (type, mode, stepN));
}

int 
MasterDof :: unpackAndUpdateUnknown (CommunicationBuffer& buff, UnknownType type, 
                   ValueModeType mode, TimeStep* stepN)
{

// result = buff.unpackDouble (&value);

 // if dof belonging to shared or remote DofManager, engng model unknowns are updated 
 // to accomodate remote contribution or "prescribed" remote values.
 // The unknown dictionary is not updated, it is engng model job to update 
 // all unknowns dictionaries.

// if (dofManager->giveParallelMode () == DofManager_shared) mode = EngngModel_Add_Mode;
// else if (dofManager->giveParallelMode () == DofManager_remote) mode = EngngModel_Set_Mode;
// else _error ("unpackAndUpdateUnknown: unknown dofManager ParallelMode");
 _error ("unpackAndUpdateUnknown: masterDof is not supposed to unpack and update");
// dofManager->giveDomain()->giveEngngModel()->
//  updateUnknownComponent(type, mode, stepN, this->giveEquationNumber(), value, updMode);
 return 0;
}

#endif

