/* $Header: /home/cvs/bp/oofem/oofemlib/src/node.C,v 1.12.4.1 2004/04/05 15:19:43 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/

/*
 The original idea for this class comes from 
  Dubois-Pelerin, Y.: "Object-Oriented  Finite Elements: Programming concepts and Implementation",
 PhD Thesis, EPFL, Lausanne, 1992.
*/

//   file NODE.CC

#include "node.h"
#include "dof.h"
#include "slavedof.h"
#include "nodload.h"
#include "timestep.h"

#include "flotarry.h"
#include "flotmtrx.h"
#include "intarray.h"
#include "debug.h"
#include "verbose.h"
#ifndef __MAKEDEPEND
#include <math.h>
#include <stdlib.h>
#endif

#ifdef __OOFEG
#include "oofeggraphiccontext.h"
#endif

Node :: Node (int n, Domain* aDomain)
      : DofManager (n,aDomain), coordinates()
   // Constructor. Creates a node with number n, belonging to aDomain.
{
  localCoordinateSystem = NULL ;
}




Node :: ~Node()
   // Destructor.
{
  delete localCoordinateSystem;
}


double  Node :: giveCoordinate (int i)
   // Returns the i-th coordinate of the receiver.
{
 if (i > coordinates.giveSize()) return 0.;
 return coordinates.at(i) ;
}


IRResultType
Node :: initializeFrom (InputRecord* ir)
   // Gets from the source line from the data file all the data of the receiver.
{
 const char *__keyword, *__proc = "initializeFrom"; // Required by IR_GIVE_FIELD macro
 IRResultType result;                               // Required by IR_GIVE_FIELD macro

 int j, size ;
 FloatArray triplets;
 // IntArray *dofIDArry;

#  ifdef VERBOSE
// VERBOSE_PRINT1("Instanciating node ",number)
#  endif

 DofManager :: initializeFrom (ir);
 IR_GIVE_FIELD (ir, coordinates, "coords"); // Macro

 // Read if available local coordinate system in this node
 triplets.resize(0);
 IR_GIVE_OPTIONAL_FIELD (ir, triplets, "lcs"); // Macro
 size=triplets.giveSize();
 if (!((size==0)||(size ==6))) 
   _warning2 ("initializeFrom: lcs in node %d is not properly defined, will be ignored", this->giveNumber());
 if (size==6) {

  double n1=0.0, n2=0.0 ;
  localCoordinateSystem = new FloatMatrix(3,3);

  for (j=1; j<= 3; j++) {
   localCoordinateSystem->at(1,j) = triplets.at(j);
   n1 += triplets.at(j)*triplets.at(j);
   localCoordinateSystem->at(2,j) = triplets.at(j+3);
   n2 += triplets.at(j+3)*triplets.at(j+3);
  }
  n1 = sqrt(n1); n2 = sqrt(n2);
  if ((n1 <= 1.e-6) || (n2 <=1.e-6))
    _error ("instanciateFrom : lcs input error");
  for (j=1; j<= 3; j++) { // normalize e1' e2'
   localCoordinateSystem->at(1,j) /= n1;
   localCoordinateSystem->at(2,j) /= n2;
  }
  // vector e3' computed from vector product of e1', e2'
  localCoordinateSystem->at(3,1) = 
   localCoordinateSystem->at(1,2)*localCoordinateSystem->at(2,3) -
   localCoordinateSystem->at(1,3)*localCoordinateSystem->at(2,2) ;
  localCoordinateSystem->at(3,2) = 
   localCoordinateSystem->at(1,3)*localCoordinateSystem->at(2,1) -
   localCoordinateSystem->at(1,1)*localCoordinateSystem->at(2,3) ;
  localCoordinateSystem->at(3,3) = 
   localCoordinateSystem->at(1,1)*localCoordinateSystem->at(2,2) -
   localCoordinateSystem->at(1,2)*localCoordinateSystem->at(2,1) ;
 }
 
 return IRRT_OK;
}



void  Node :: printYourself ()
   // Prints the receiver on screen.
{
   int    i ;
   double x,y ;

   x = this->giveCoordinate(1) ;
   y = this->giveCoordinate(2) ;
   printf ("Node %d    coord : x %f  y %f\n",number,x,y) ;
   for (i=0 ; i<numberOfDofs ; i++) {
      if (dofArray[i])
  dofArray[i] -> printYourself() ;
      else
  printf ("dof %d is nil \n",i+1) ;}
  loadArray.printYourself() ;
   printf ("\n") ;
}


void  Node :: updateYourself (TimeStep* tStep)
   // Updates the receiver at end of step.
{
   int i,ic ;

#  ifdef VERBOSE
// VERBOSE_PRINT1 ("Updating node ",number)
#  endif

  fMode mode = domain->giveEngngModel()-> giveFormulation() ;
   
   DofManager::updateYourself (tStep);

  for (i=1 ; i<=numberOfDofs ; i++) {
     if (mode == AL) {      // updated Lagrange
       ic = domain ->  giveCorrespondingCoordinateIndex (i);
       if (ic != 0) { // && (this->giveDof(i)->giveUnknownType() == DisplacementVector))   {
     coordinates.at(ic) += 
      this->giveDof(i)->giveUnknown (DisplacementVector,VM_Incremental,tStep);
       }
     }
   }
 }


double Node :: giveUpdatedCoordinate (int ic ,TimeStep* tStep, UnknownType type, double scale )
//
// returns coordinate + scale * displacement
// displacement is of updMode (UpdateMode) type
//
{
 int i, j;
 FloatMatrix *T;
 
 if ((ic < 1) || (ic > 3)) {
  _error ("giveUpdatedCoordinate: Can't return non-existing coordinate (index not in range 1..3)");
  return 0.;
 }

 if (tStep->isTheCurrentTimeStep ()) {
  double coordinate = this->giveCoordinate (ic);
  if (!this->hasLocalCS ()) {
   // this has no local cs.
   for (i=1 ; i<=numberOfDofs ; i++) {
    j = domain ->  giveCorrespondingCoordinateIndex (i);
    if ((j != 0) && (j == ic))   {
     coordinate += 
      scale * this->giveDof(i)->giveUnknown (type,VM_Total,tStep);
     break;
    }
   }
  } else {
   //
   // this has local cs.
   // We must perform transformation of displacements DOFs
   // in to global c.s and then to add them to global coordinates.
   //
   T = this->giveLocalCoordinateTriplet() ;
   FloatArray displacements (3) ;
   for (i=1 ; i<= 3; i++) displacements.at(i) = 0.;
   for (i=1 ; i<=numberOfDofs; i++) {
    j = domain ->  giveCorrespondingCoordinateIndex (i);
    if (j) // && (this->giveDof(i)->giveUnknownType()==DisplacementVector))
     displacements.at(j) = scale * this->giveDof(i)->
      giveUnknown (type,VM_Total,tStep);
   }
   // perform transformation for desired displacement
   for (i=1 ; i<= 3; i++) 
    coordinate += displacements.at(i) * T-> at(i,ic);
  }
  return coordinate;
 } else { _error("Can't return updatedCoordinate for non-current timestep");}
 return 0.;
}

int
Node::checkConsistency () {
/*
  Checks internal data consistency in node. 
  Current implementation checks (when receiver has slave dofs) if receiver has the same 
  coordinate system as master dofManager of slave dof.
*/
 int result = 1;
 int ndofs = this->giveNumberOfDofs ();
 int i, nslaves=0;

 result = result && DofManager::checkConsistency();

 for (i=1; i<=ndofs; i++) 
  if (this->giveDof (i)->giveClassID() == SlaveDofClass) nslaves ++;

 if (nslaves == 0) return result;  // return o.k. if no slaves exists
 
 IntArray masterDofManagers (nslaves);
 int numberOfMDM = 0;  // counter of diferent master dofManagers
 int j, master, alreadyFound = 0;
 Dof* idof;
 Node* masterNode;

 for (i=1; i<=ndofs; i++) {
  if ((idof = this->giveDof (i))->giveClassID() == SlaveDofClass) {
   alreadyFound  = 0;
   master = ((SlaveDof*) idof)->giveMasterDofManagerNum();
   for (j=1; j<= numberOfMDM; j++) 
    if (masterDofManagers.at(j) == master) {
     alreadyFound = 1;
     break;
    }
   if (alreadyFound == 0) {
    // check master for same coordinate system
    // first mark master as checked
    numberOfMDM ++;
    masterDofManagers.at(numberOfMDM) = master;
    // compare coordinate systems
    masterNode = dynamic_cast<Node*>(domain->giveDofManager(master));
    if (masterNode) {

     FloatMatrix *thisLcs, *masterLcs;
     thisLcs = this->giveLocalCoordinateTriplet();
     masterLcs = masterNode->giveLocalCoordinateTriplet();
     int k,l;

     if ((this->hasLocalCS()) && (masterNode->hasLocalCS())) {
      for (k=1; k<=3; k++)
       for (l=1; l<=3; l++)
        if (fabs(thisLcs->at(k,l)-masterLcs->at(k,l)) > 1.e-4) {
         _warning2 ("checkConsistency: different lcs for master/slave nodes", 1);
         return 0;
        }
     } else if (this->hasLocalCS()) {
      for (k=1; k<=3; k++)
       for (l=1; l<=3; l++)
        if (fabs(thisLcs->at(k,l)-(k==l)) > 1.e-4) {
         _warning2 ("checkConsistency: different lcs for master/slave nodes", 1);
         return 0;
        }
     } else if (masterNode->hasLocalCS()) {
      for (k=1; k<=3; k++)
       for (l=1; l<=3; l++)
        if (fabs(masterLcs->at(k,l)-(k==l)) > 1.e-4) {
         _warning2 ("checkConsistency: different lcs for master/slave nodes", 1);
         return 0;
        }
     }
    } else {
     _warning2 ("checkConsistency: master dofManager is not compatible", 1);
     return 0;
    }
   }
  }
 }
 return result;
}


void
Node::computeDofTransformation (FloatMatrix& answer, const IntArray* dofIDArry, DofManTrasfType mode)
{
 // computes trasformation matrix of receiver.
 // transformation should include trasformation from global cs to nodal cs,
 // as well as further necessary transformations (for example in case 
 // rigid arms this must include transformation to master dofs).

 this->computeGNDofTransformation (answer, dofIDArry);
 if (mode == _toGlobalCS) {
  FloatMatrix answert;
  answert.beTranspositionOf (answer);
  answer = answert;
 }
}

void
Node::computeLoadTransformation (FloatMatrix& answer, const IntArray* dofIDArry, DofManTrasfType mode)
{
 // computes trasformation matrix of receiver.
 // transformation should include trasformation from global cs to nodal cs,
 // as well as further necessary transformations (for example in case 
 // rigid arms this must include transformation to master dofs).

 this->computeGNDofTransformation (answer, dofIDArry);
 if (mode == _toGlobalCS) {
  FloatMatrix answert;
  answert.beTranspositionOf (answer);
  answer = answert;
 }
}

void
Node::computeGNDofTransformation (FloatMatrix& answer, const IntArray* map) 
{
 //
 // computes transfromation of receiver from global cs to nodal (user-defined) cs.
 // Note: implementation rely on D_u, D_v and D_w (R_u, R_v, R_w) order in cltypes.h
 // file. Do not change their order and do not insert any values between these values.
 // 
 //

 int i,j;
 DofIDItem id, id2;

 if (localCoordinateSystem == NULL) {
  // localCoordinateSystem same as global c.s.
  int size;
  if (map == NULL) size = numberOfDofs; else size = map->giveSize();
  answer.resize (size, size);
  answer.zero();
  for (i=1; i<=size; i++) answer.at(i,i) = 1.0;

 } else {
  if (map == NULL) {
   // response for all local dofs is computed
   
   answer.resize (numberOfDofs, numberOfDofs);
   answer.zero();
   
   for (i=1; i<=numberOfDofs; i++) {
    // test for vector quantities
    switch (id = giveDof(i)->giveDofID()) {
    case D_u:
    case D_v:
    case D_w:
     for (j=1; j<= numberOfDofs; j++) {
      id2 = giveDof(j)->giveDofID();
      if ((id2 == D_u) || (id2 == D_v) || (id2 == D_w))
       answer.at(i,j) = localCoordinateSystem->at((int)(id)-(int)(D_u)+1, 
                             (int)(id2)-(int)(D_u)+1);
     }
     break;
     
    case R_u:
    case R_v:
    case R_w:
     for (j=1; j<= numberOfDofs; j++) {
      id2 = giveDof(j)->giveDofID();
      if ((id2 == R_u) || (id2 == R_v) || (id2 == R_w))
       answer.at(i,j) = localCoordinateSystem->at((int)(id)-(int)(R_u)+1, 
                             (int)(id2)-(int)(R_u)+1);
     }
     break;
     
    case T_f:
    case P_f:
     // scalar quantities
     answer.at(i,i) = 1.0;
     break;
     
    default:
     _error ("computeGNTransformation: unknown dofID");
    }
   }
  } else { // end if (map == NULL) 
   // map is provided -> assemble for requested dofs
   int size = map->giveSize();
   answer.resize (size, size);
   answer.zero();
   
   for (i=1; i<=size; i++) {
    // test for vector quantities
    switch (id = (DofIDItem) map->at(i)) {
    case D_u:
    case D_v:
    case D_w:
     for (j=1; j<= size; j++) {
      id2 = (DofIDItem) map->at(j);
      if ((id2 == D_u) || (id2 == D_v) || (id2 == D_w))
       answer.at(i,j) = localCoordinateSystem->at((int)(id)-(int)(D_u)+1, (int)(id2)-(int)(D_u)+1);
     }
     break;
     
    case R_u:
    case R_v:
    case R_w:
     for (j=1; j<= size; j++) {
      id2 = (DofIDItem) map->at(j);
      if ((id2 == R_u) || (id2 == R_v) || (id2 == R_w))
       answer.at(i,j) = localCoordinateSystem->at((int)(id)-(int)(R_u)+1, (int)(id2)-(int)(R_u)+1);
     }
     break;
     
    case T_f:
    case P_f:
     // scalar quantities
     answer.at(i,i) = 1.0;
     break;
     
    default:
     _error ("computeGNTransformation: unknown dofID");
    }
    
   }
  } // end map is provided -> assemble for requested dofs
 } // end localCoordinateSystem defined
}





#ifdef __OOFEG
void Node :: drawYourself (oofegGraphicContext& gc)
//
// draws graphics representation of receiver 
//
{
  GraphicObj *go;
  OGC_PlotModeType mode = gc.giveIntVarPlotMode();

  if ((mode == OGC_nodeGeometry) || (mode == OGC_nodeAnnotation)) {
     WCRec p[1];   /* point */
     p[0].x = (FPNum) this->giveCoordinate(1);
     p[0].y = (FPNum) this->giveCoordinate(2);
     p[0].z = (FPNum) this->giveCoordinate(3);
  
     EASValsSetLayer(OOFEG_NODE_ANNOTATION_LAYER);
     EASValsSetMType(FILLED_CIRCLE_MARKER);
     EASValsSetColor(gc.getNodeColor());
     EASValsSetMSize(8);
     go = CreateMarker3D(p);
     EGWithMaskChangeAttributes(COLOR_MASK | LAYER_MASK | MTYPE_MASK | MSIZE_MASK, go);
     EMAddGraphicsToModel(ESIModel(), go);

  } 

  if (mode == OGC_nodeAnnotation) {
    char num[6];
    WCRec p[1];   /* point */
    EASValsSetColor(gc.getNodeColor());
    EASValsSetLayer(OOFEG_NODE_ANNOTATION_LAYER);
    p[0].x = (FPNum) this->giveCoordinate(1);
    p[0].y = (FPNum) this->giveCoordinate(2);
    p[0].z = (FPNum) this->giveCoordinate(3);
    sprintf(num,"%d",this->giveNumber());
    go = CreateAnnText3D (p,num);
    EGWithMaskChangeAttributes(COLOR_MASK | LAYER_MASK, go);
    EMAddGraphicsToModel(ESIModel(), go);

  } else if (mode == OGC_appliedPrimaryBc) {

  int i,hasDisplSupport[3],hasRotSupport[3],hasAny = 0;
  TimeStep *tStep = domain->giveEngngModel()->giveCurrentStep();
  WCRec pp[2];

  for (i=0; i< 3; i++) {
   hasDisplSupport[i]=0;
   hasRotSupport[i]=0;
  }

  for (i = 1; i<= numberOfDofs; i++) {
   if (this->giveDof(i)->hasBc (tStep)) {
    hasAny = 1;
    switch (giveDof(i)->giveDofID()) {
    case D_u: hasDisplSupport[0] = 1; break;
    case D_v: hasDisplSupport[1] = 1; break;
    case D_w: hasDisplSupport[2] = 1; break;
    case R_u: hasRotSupport[0] = 1; break;
    case R_v: hasRotSupport[1] = 1; break;
    case R_w: hasRotSupport[2] = 1; break;
    default: break;
    }
   }
  }
  
  if (hasAny == 0 ) return;
  
  EASValsSetColor(gc.getBcIcColor());
  EASValsSetLayer(OOFEG_BCIC_ANNOTATION_LAYER);
  pp[0].x = (FPNum) this->giveCoordinate(1);
  pp[0].y = (FPNum) this->giveCoordinate(2);
  pp[0].z = (FPNum) this->giveCoordinate(3);
  
  for (i = 0; i< 3; i++) {
   if (hasDisplSupport[i] || hasRotSupport[i]) {
    pp[1].x = 0.; pp[1].y = 0.; pp[1].z = 0.;

    if (!this->hasLocalCS ()) {
     if (i == 0) pp[1].x = 1.0;
     if (i == 1) pp[1].y = 1.0;
     if (i == 2) pp[1].z = 1.0;
    } else {
     FloatMatrix *T = this->giveLocalCoordinateTriplet() ;;
     pp[1].x = T->at(i+1,1);
     pp[1].y = T->at(i+1,2);
     pp[1].z = T->at(i+1,3);
    }    
    
    
    if (hasDisplSupport[i] && hasRotSupport[i]) 
     EASValsSetVecMType(TRIPLE_ARROW_VECMARKER);
    if (hasDisplSupport[i]) EASValsSetVecMType(ARROW_VECMARKER);
    if (hasRotSupport[i]) EASValsSetVecMType(DOUBLE_ARROW_VECMARKER);
    go = CreateVecMarker3D(pp);
    EGWithMaskChangeAttributes(COLOR_MASK | LAYER_MASK | VECMTYPE_MASK , go);
    EMAddGraphicsToModel(ESIModel(), go);
   }
  }
 }
 
}

#endif
