/* $Header: /home/cvs/bp/oofem/sm/src/cct.C,v 1.5 2003/04/06 14:08:30 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/

//   file CCTPlate.CC

#include "cct.h"
#include "node.h"
#include "material.h"
#include "crosssection.h"
#include "structuralms.h"
#include "gausspnt.h"
#include "gaussintegrationrule.h"
#include "flotmtrx.h"
#include "flotarry.h"
#include "intarray.h"
#include "domain.h"
#include "verbose.h"
#include "engngm.h"
#include "load.h"
#include "structuralcrosssection.h"
#include "structuralmaterial.h"
#ifndef __MAKEDEPEND
#include <math.h>
#include <stdio.h>
#endif

#ifdef __OOFEG
#include "oofeggraphiccontext.h"
#include "conTable.h"
#endif

CCTPlate :: CCTPlate (int n, Domain* aDomain)
          : NLStructuralElement (n,aDomain), LayeredCrossSectionInterface(), ZZNodalRecoveryModelInterface(), 
      NodalAveragingRecoveryModelInterface(), SPRNodalRecoveryModelInterface()
// Constructor.
{
   numberOfNodes  = 3 ;
  area = -1;
}

Interface* 
CCTPlate :: giveInterface (InterfaceType interface)
{
 if (interface == LayeredCrossSectionInterfaceType) return (LayeredCrossSectionInterface*) this;
 else if (interface == ZZNodalRecoveryModelInterfaceType) return (ZZNodalRecoveryModelInterface*) this;
 else if (interface == NodalAveragingRecoveryModelInterfaceType) return (NodalAveragingRecoveryModelInterface*) this;
 else if (interface == SPRNodalRecoveryModelInterfaceType) return (SPRNodalRecoveryModelInterface*) this;
 return NULL;
}

double
CCTPlate :: giveArea ()
// returns the area occupied by the receiver
{
 if (area > 0) return area;  // check if previously computed

   Node    *node1,*node2,*node3;
   double  x1,x2,x3,y1,y2,y3;

   node1 = this -> giveNode(1) ;
   node2 = this -> giveNode(2) ;
   node3 = this -> giveNode(3) ;

   x1 = node1 -> giveCoordinate(1) ;
   x2 = node2 -> giveCoordinate(1) ;
   x3 = node3 -> giveCoordinate(1) ;

   y1 = node1 -> giveCoordinate(2) ;
   y2 = node2 -> giveCoordinate(2) ;
   y3 = node3 -> giveCoordinate(2) ;

   return (area = 0.5*(x2*y3+x1*y2+y1*x3-x2*y1-x3*y2-x1*y3)) ;
 
}

void
CCTPlate :: computeBmatrixAt (GaussPoint *aGaussPoint, FloatMatrix& answer, int li, int ui)
// Returns the [5x9] strain-displacement matrix {B} of the receiver,
// evaluated at aGaussPoint.
{
 //FloatMatrix *answer ;
 double x1,x2,x3,y1,y2,y3,area,b1,b2,b3,c1,c2,c3,l1,l2,l3;
 
   x1 = this -> giveNode(1) -> giveCoordinate(1);
   x2 = this -> giveNode(2) -> giveCoordinate(1);
   x3 = this -> giveNode(3) -> giveCoordinate(1);
 
   y1 = this -> giveNode(1) -> giveCoordinate(2);
   y2 = this -> giveNode(2) -> giveCoordinate(2);
   y3 = this -> giveNode(3) -> giveCoordinate(2);

   b1 = y2-y3;
   b2 = y3-y1;
   b3 = y1-y2;

   c1 = x3-x2;
   c2 = x1-x3;
   c3 = x2-x1;

   l1 = 1./3.;
   l2 = 1./3.;
   l3 = 1./3.;

   area = 0.5*(x2*y3+x1*y2+y1*x3-x2*y1-x3*y2-x1*y3) ;

   //answer = new FloatMatrix(5,9);
 answer.resize (5,9);
 answer.zero();

   answer.at(1,3) = b1;
   answer.at(1,6) = b2;
   answer.at(1,9) = b3;

   answer.at(2,2) = -c1 ;
   answer.at(2,5) = -c2 ;
   answer.at(2,8) = -c3 ;

   answer.at(3,2) = -b1 ;
   answer.at(3,3) =  c1 ;
   answer.at(3,5) = -b2 ;
   answer.at(3,6) =  c2 ;
   answer.at(3,8) = -b3 ;
   answer.at(3,9) =  c3 ;
   
   answer.at(4,1) =  b1 ;
   answer.at(4,2) = (-b1*b3*l2+b1*b2*l3)*0.5 ;
   answer.at(4,3) = (-b1*c3*l2-b2*c3*l1+b3*c2*l1+b1*c2*l3)*0.5+l1*2.*area ;
   answer.at(4,4) =  b2 ;
   answer.at(4,5) = (-b2*b1*l3+b2*b3*l1)*0.5 ;
   answer.at(4,6) = (-b2*c1*l3-b3*c1*l2+b1*c3*l2+b2*c3*l1)*0.5+l2*2.*area ;
   answer.at(4,7) =  b3 ;
   answer.at(4,8) = (-b3*b2*l1+b3*b1*l2)*0.5 ;
   answer.at(4,9) = (-b3*c2*l1-b1*c2*l3+b2*c1*l3+b3*c1*l2)*0.5+l3*2.*area ;

   answer.at(5,1) =  c1 ;
   answer.at(5,2) = (-b3*c1*l2-b3*c2*l1+b2*c3*l1+b2*c1*l3)*0.5-l1*2.*area ;
   answer.at(5,3) = (-c1*c3*l2+c1*c2*l3)*0.5 ;
   answer.at(5,4) =  c2 ;
   answer.at(5,5) = (-b1*c2*l3-b1*c3*l2+b3*c1*l2+b3*c2*l1)*0.5-l2*2.*area ;
   answer.at(5,6) = (-c2*c1*l3+c2*c3*l1)*0.5 ;
   answer.at(5,7) =  c3 ;
   answer.at(5,8) = (-b2*c3*l1-b2*c1*l3+b1*c2*l3+b1*c3*l2)*0.5-l3*2.*area ;
   answer.at(5,9) = (-c3*c2*l1+c3*c1*l2)*0.5 ;

   answer.times(1./(2.*area));
   return  ;
}
   


void  CCTPlate :: computeGaussPoints ()
   // Sets up the array containing the four Gauss points of the receiver.
{
   numberOfIntegrationRules = 1 ;
  integrationRulesArray = new IntegrationRule*;
  integrationRulesArray[0] = new GaussIntegrationRule (1, domain, 1, 5);
  integrationRulesArray[0]->setUpIntegrationPoints (_Triangle, numberOfGaussPoints, this, _2dPlate);
}



void
CCTPlate :: computeNmatrixAt (GaussPoint* aGaussPoint, FloatMatrix& answer) 
// Returns the displacement interpolation matrix {N} of the receiver,
// evaluated at aGaussPoint.
{
   double x1,x2,x3,y1,y2,y3,l1,l2,l3,b1,b2,b3,c1,c2,c3;
   // FloatMatrix* answer;

   l1 = aGaussPoint -> giveCoordinate(1);
   l2 = aGaussPoint -> giveCoordinate(2);
  l3 = 1.0 - l1 - l2;

   x1 = this -> giveNode(1) -> giveCoordinate(1);
   x2 = this -> giveNode(2) -> giveCoordinate(1);
   x3 = this -> giveNode(3) -> giveCoordinate(1);
 
   y1 = this -> giveNode(1) -> giveCoordinate(2);
   y2 = this -> giveNode(2) -> giveCoordinate(2);
   y3 = this -> giveNode(3) -> giveCoordinate(2);

   b1 = y2-y3;
   b2 = y3-y1;
   b3 = y1-y2;

   c1 = x3-x2;
   c2 = x1-x3;
   c3 = x2-x1;

   answer.resize(3,9);
  answer.zero();

   answer.at(1,1) = l1;
   answer.at(1,2) = (l1*l2*b3-l3*l1*b2)*0.5;
   answer.at(1,3) = (l1*l2*c3-l3*l1*c2)*0.5;
   answer.at(1,4) = l2;
   answer.at(1,5) = (l2*l3*b1-l1*l2*b3)*0.5;
   answer.at(1,6) = (l2*l3*c1-l1*l2*c3)*0.5; 
   answer.at(1,7) = l3;
   answer.at(1,8) = (l3*l1*b2-l2*l3*b1)*0.5;
   answer.at(1,9) = (l3*l1*c2-l2*l3*c1)*0.5;

   answer.at(2,2) = l1;
   answer.at(2,5) = l2;
   answer.at(2,8) = l3;

   answer.at(3,3) = l1;
   answer.at(3,6) = l2;
   answer.at(3,9) = l3;

   return ;
}

IRResultType
CCTPlate :: initializeFrom (InputRecord* ir)
{
 const char *__keyword, *__proc = "initializeFrom"; // Required by IR_GIVE_FIELD macro
 IRResultType result;                               // Required by IR_GIVE_FIELD macro

 this->NLStructuralElement :: initializeFrom (ir);
 numberOfGaussPoints = 1;
 IR_GIVE_OPTIONAL_FIELD (ir, numberOfGaussPoints, "nip"); // Macro
 if (numberOfGaussPoints != 1) numberOfGaussPoints = 1;

 this -> computeGaussPoints();
 return IRRT_OK;
}

double
CCTPlate :: computeVolumeAround (GaussPoint* aGaussPoint)
   // Returns the portion of the receiver which is attached to aGaussPoint.
{
 double area,weight ;
 
 weight  = aGaussPoint -> giveWeight() ;
 area    = this -> giveArea();
 
 return 2.0*area * weight ;
}

void
CCTPlate :: computeLumpedMassMatrix (FloatMatrix& answer, TimeStep* tStep)
// Returns the lumped mass matrix of the receiver.
{
   GaussPoint* gp;
   double dV,mss1 ;
 
   answer.resize (9,9); answer.zero();

   gp                 = integrationRulesArray[0]->getIntegrationPoint(0);
 
   dV = this->computeVolumeAround(gp) ;
   mss1 = dV * this->giveMaterial()->give('d')*
     this -> giveCrossSection()->give(THICKNESS) / 3. ;
 
   answer.at(1,1) = mss1 ;
   answer.at(4,4) = mss1 ;
   answer.at(7,7) = mss1 ;
   
  //this -> giveRotationMatrix () ;
  //if (rotationMatrix) answer.rotatedWith(*rotationMatrix) ;

   return  ;
}

void
CCTPlate :: computeBodyLoadVectorAt (FloatArray& answer, Load* forLoad, TimeStep* stepN, ValueModeType mode)
// Computes numerically the load vector of the receiver due to the body
// loads, at stepN.
{
   double      dV, load;
   GaussPoint* gp = NULL;
   FloatArray  f,*coord1 ;


  forLoad -> computeComponentArrayAt(f, stepN, mode);
  f.times(this->giveMaterial() -> give('d'));
  
  if (f.giveSize()==0) {
  answer.resize (0); return ;                              // nil resultant 
  } else {
  coord1          = new FloatArray(2) ;
  coord1 -> at(1) = 0 ;
  coord1 -> at(2) = 0 ;
  gp              = integrationRulesArray[0]->getIntegrationPoint(0);
  
  dV = this->computeVolumeAround(gp);             // Area
  answer.resize (9) ;
  answer.zero();
  
  load=f.at(1) * dV * this -> giveCrossSection()->give(THICKNESS) / 3.0;
  answer.at(1)=load ;
  answer.at(4)=load ;
  answer.at(7)=load ;
  
  //delete f ;
  return  ;
  }
}

FloatArray*   
CCTPlate :: ComputeMidPlaneNormal (GaussPoint*)
//
// returns normal vector to midPlane in GaussPoinr gp of receiver
//
{
 Node *n1, *n2, *n3;
 FloatArray *answer, *u, *v;
 int i;
 
 u = new FloatArray (3);
 v = new FloatArray (3);
 answer = new FloatArray (3);
 
 n1 = this->giveNode (1);
 n2 = this->giveNode (2);
 n3 = this->giveNode (3);

 for (i=1; i<= 3; i++) {
  u->at(i)=n2->giveCoordinate(i) - n1->giveCoordinate(i);
  v->at(i)=n3->giveCoordinate(i) - n1->giveCoordinate(i);
 }
 answer = u -> VectorProduct (v);
 answer -> normalize();
 
 delete u; delete v;
 return answer;

}


/*
int
CCTPlate :: computeGtoNRotationMatrix (FloatMatrix& answer)
// returns transformation matrix from global coordinate set to
// nodal coordinate set
// return NULL if no trasformation necessary
{
 FloatMatrix *triplet;
 int i,flag=0,ii;
 
 for (i=1; i<= numberOfNodes; i++)
  flag += this->giveNode(i)->hasLocalCS ();
 if (flag == 0) {answer.beEmptyMtrx(); return 0 ;}
 
 answer.resize (9,9); answer.zero();
 // loop over nodes
 for (i=1; i<= numberOfNodes; i++) {
  ii = (i-1)*3+1 ;
  if (this->giveNode(i)->hasLocalCS ()) {
   triplet = this->giveNode(i)->giveLocalCoordinateTriplet();
   answer.at(ii,ii)     = 1.0 ;
   answer.at(ii,ii+1)   = 0.0 ;
   answer.at(ii,ii+2)   = 0.0 ;
   answer.at(ii+1,ii)   = 0.0 ;
   answer.at(ii+1,ii+1) = triplet->at(1,1) ;
   answer.at(ii+1,ii+2) = triplet->at(1,2) ;
   answer.at(ii+2,ii)   = 0.0 ;
   answer.at(ii+2,ii+1) = triplet->at(2,1) ;
   answer.at(ii+2,ii+2) = triplet->at(2,2) ;
   
  } else {
   // no transformation - unit matrix as 
   // transformation submatrix for node i
   answer.at(ii,ii)     = 1.0;
   answer.at(ii+1,ii+1) = 1.0;
   answer.at(ii+2,ii+2) = 1.0;
  }
 }
 
 return 1;
}
*/

void
CCTPlate:: computeStrainVectorInLayer (FloatArray &answer, GaussPoint* masterGp,
                    GaussPoint* slaveGp, TimeStep* tStep)
//
// returns full 3d strain vector of given layer (whose z-coordinate from center-line is
// stored in slaveGp) for given tStep
// 
{
 FloatArray masterGpStrain;
 double layerZeta, layerZCoord, top, bottom;

 this->computeStrainVector (masterGpStrain, masterGp, tStep);
 top    = masterGp-> giveElement()->giveCrossSection()->give(TOPZCOORD);
 bottom = masterGp-> giveElement()->giveCrossSection()->give(BOTTOMZCOORD);
 layerZeta = slaveGp -> giveCoordinate (1);
 layerZCoord = 0.5*((1.- layerZeta)*bottom + (1.+ layerZeta)*top);

 answer.resize (6);   // {Exx,Eyy,Ezz,GMyz,GMzx,GMxy}
 answer.zero();

 answer.at(1) = masterGpStrain.at(1) * layerZCoord ;
 answer.at(2) = masterGpStrain.at(2) * layerZCoord ;
 answer.at(6) = masterGpStrain.at(3) * layerZCoord ;
 answer.at(4) = masterGpStrain.at(5);
 answer.at(5) = masterGpStrain.at(4);

 return ;

}

void
CCTPlate ::   giveNodeDofIDMask  (int inode, IntArray &answer) const {
// returns DofId mask array for inode element node.
// DofId mask array determines the dof ordering requsted from node.
// DofId mask array contains the DofID constants (defined in cltypes.h)
// describing physical meaning of particular DOFs.
 //IntArray* answer = new IntArray (3);
 answer.resize (3);

 answer.at(1) = D_w;
 answer.at(2) = R_u;
 answer.at(3) = R_v;

 return ;
}


double
CCTPlate :: giveCharacteristicLenght (GaussPoint* gp, const FloatArray& normalToCrackPlane)
//
// returns receivers characteristic length in gp (for some material models)
// for crack formed in plane with normal normalToCrackPlane.
//
{
 return this -> giveLenghtInDir (normalToCrackPlane);
}



/*
void
CCTPlate :: computeTemperatureStrainVectorAt (FloatArray& answer, GaussPoint* gp, TimeStep* stepN, ValueModeType mode)
{
  // computes temperature strain vector of the receiver
 StructuralMaterial * mat = (StructuralMaterial*) this->giveMaterial();
 StructuralCrossSection* cs = (StructuralCrossSection*) this->giveCrossSection();
 FloatArray  et, e0;
 double thick;

 if (this -> giveBodyLoadArray() -> isEmpty()) {answer.resize(0); return;}
   
 this -> computeResultingIPTemperatureAt (et, stepN, gp, mode);
 if (et.giveSize() == 0) {answer.resize(0); return;}
 if (et.giveSize() < 1) {
   _error ("computeTemperatureStrainVectorAt - Bad format of TemperatureLoad");
   exit (1);
 }
 mat->giveThermalDilatationVector (e0, gp,stepN);

 if (e0.giveSize()) {
   if (et.giveSize() > 1) {
   answer.resize (5);
   answer.zero();

   thick = cs->give(THICKNESS);
   answer.at(1) = e0.at(1) * et.at(2)/ thick;   // kappa_x
   answer.at(2) = e0.at(2) * et.at(2)/ thick;   // kappa_y

   }
 }
 //delete et;
 //delete e0;
 
 return ;
}
*/



int
CCTPlate :: computeGlobalCoordinates (FloatArray& answer, const FloatArray& lcoords) 
{
 double l1,l2,l3;
   
 l1 = lcoords.at(1);
 l2 = lcoords.at(2);
 l3 = 1.0 - l1 - l2;
  
 answer.resize (2);
 answer.at(1) = l1*this->giveNode(1)->giveCoordinate(1)+l2*this->giveNode(2)->giveCoordinate(1) +
  l3 * this->giveNode(3)->giveCoordinate(1);
 answer.at(2) = l1*this->giveNode(1)->giveCoordinate(2)+l2*this->giveNode(2)->giveCoordinate(2) +
  l3 * this->giveNode(3)->giveCoordinate(2);

 return 1;
}

int 
CCTPlate :: ZZNodalRecoveryMI_giveDofManRecordSize(InternalStateType type)
{
 if ((type == IST_ForceTensor) || (type == IST_MomentumTensor)) return 5;

 return 0;
}


void
CCTPlate :: ZZNodalRecoveryMI_ComputeEstimatedInterpolationMtrx  
(FloatMatrix& answer, GaussPoint* aGaussPoint, InternalStateType type)
{
  // evaluates N matrix (interpolation estimated stress matrix)
  // according to Zienkiewicz & Zhu paper
  // N(nsigma, nsigma*nnodes)
  // Definition : sigmaVector = N * nodalSigmaVector
   double l1,l2,l3;

   l1 = aGaussPoint -> giveCoordinate(1);
   l2 = aGaussPoint -> giveCoordinate(2);
   l3 = 1.0 - l1 - l2;

  if ((type == IST_ForceTensor) || (type == IST_MomentumTensor)) answer.resize(1,3) ;
  else return;
  
  answer.zero();
  answer.at(1,1) = l1 ;
  answer.at(1,2) = l2 ;
  answer.at(1,3) = l3 ;

   return ;
}

void 
CCTPlate::NodalAveragingRecoveryMI_computeNodalValue (FloatArray& answer, int node,
                           InternalStateType type, TimeStep* tStep)
{
 GaussPoint* gp;
 if ((type == IST_ForceTensor) || (type == IST_MomentumTensor)) {
  gp = integrationRulesArray[0]-> getIntegrationPoint(0) ;
  answer = ((StructuralMaterialStatus*) this->giveMaterial()->giveStatus(gp)) -> giveStressVector();
 } else answer.resize(0);
}

void 
CCTPlate::NodalAveragingRecoveryMI_computeSideValue (FloatArray& answer, int side,
                              InternalStateType type, TimeStep* tStep)
{
 answer.resize(0);
}



void 
CCTPlate::SPRNodalRecoveryMI_giveSPRAssemblyPoints (IntArray& pap)
{
 pap.resize(3);
 pap.at(1) = this->giveNode(1)->giveNumber();
 pap.at(2) = this->giveNode(2)->giveNumber();
 pap.at(3) = this->giveNode(3)->giveNumber();
}

void
CCTPlate::SPRNodalRecoveryMI_giveDofMansDeterminedByPatch (IntArray& answer, int pap)
{
 answer.resize(1);
 if ((pap == this->giveNode(1)->giveNumber()) ||
   (pap == this->giveNode(2)->giveNumber()) ||
   (pap == this->giveNode(3)->giveNumber())) answer.at(1) = pap;
 else _error ("SPRNodalRecoveryMI_giveDofMansDeterminedByPatch: node unknown");
}

int
CCTPlate::SPRNodalRecoveryMI_giveNumberOfIP ()
{ return 1; }


void
CCTPlate::SPRNodalRecoveryMI_computeIPGlobalCoordinates (FloatArray& coords, GaussPoint* gp)
{
 if(gp == integrationRulesArray[0]-> getIntegrationPoint(0)) {
  this->computeGlobalCoordinates(coords, *gp->giveCoordinates());
 } else {
  _error ("SPRNodalRecoveryMI_computeIPGlobalCoordinates: unsupported ip num");
 }
}

SPRPatchType
CCTPlate::SPRNodalRecoveryMI_givePatchType()
{
 return SPRPatchType_2dxy;
}

int 
CCTPlate::giveIPValue (FloatArray& answer, GaussPoint* aGaussPoint, InternalStateType type, TimeStep* atTime)
{


 if ((type == IST_ForceTensor) || (type == IST_MomentumTensor)) {
  answer = ((StructuralMaterialStatus*) this->giveMaterial()->giveStatus(aGaussPoint)) -> giveStressVector();
  return 1;
 } else if ((type == IST_CurvatureTensor) || (type == IST_StrainTensor)) {
  answer = ((StructuralMaterialStatus*) this->giveMaterial()->giveStatus(aGaussPoint)) -> giveStrainVector();
  return 1;
 } else { answer.resize(0); return 0;}
}



#ifdef __OOFEG
void
CCTPlate  :: drawRawGeometry (oofegGraphicContext &gc)
{
 WCRec p[3];
 GraphicObj *go;

 if (!gc.testElementGraphicActivity(this)) return; 
 
 EASValsSetLineWidth(OOFEG_RAW_GEOMETRY_WIDTH);
 EASValsSetColor(gc.getElementColor());
  EASValsSetEdgeColor(gc.getElementEdgeColor());
  EASValsSetEdgeFlag(TRUE);
 EASValsSetLayer(OOFEG_RAW_GEOMETRY_LAYER);
 p[0].x = (FPNum) this->giveNode(1)->giveCoordinate(1);
 p[0].y = (FPNum) this->giveNode(1)->giveCoordinate(2);
 p[0].z = (FPNum) this->giveNode(1)->giveCoordinate(3);
 p[1].x = (FPNum) this->giveNode(2)->giveCoordinate(1);
 p[1].y = (FPNum) this->giveNode(2)->giveCoordinate(2);
 p[1].z = (FPNum) this->giveNode(2)->giveCoordinate(3);
 p[2].x = (FPNum) this->giveNode(3)->giveCoordinate(1);
 p[2].y = (FPNum) this->giveNode(3)->giveCoordinate(2);
 p[2].z = (FPNum) this->giveNode(3)->giveCoordinate(3);
   
 go =  CreateTriangle3D(p);
 EGWithMaskChangeAttributes(WIDTH_MASK | COLOR_MASK | EDGE_COLOR_MASK | EDGE_FLAG_MASK | LAYER_MASK, go);
 EGAttachObject(go, (EObjectP) this);
 EMAddGraphicsToModel(ESIModel(), go);
}

void
CCTPlate  :: drawDeformedGeometry (oofegGraphicContext &gc, UnknownType type)
{
 WCRec p[3];
 GraphicObj *go;
 TimeStep *tStep = domain->giveEngngModel()->giveCurrentStep();
 double defScale = gc.getDefScale();

 if (!gc.testElementGraphicActivity(this)) return; 

 EASValsSetLineWidth(OOFEG_DEFORMED_GEOMETRY_WIDTH);
 EASValsSetColor(gc.getDeformedElementColor());
  EASValsSetEdgeColor(gc.getElementEdgeColor());
  EASValsSetEdgeFlag(TRUE);

 EASValsSetLayer(OOFEG_DEFORMED_GEOMETRY_LAYER);
 p[0].x = (FPNum) this->giveNode(1)->giveUpdatedCoordinate(1,tStep,type,defScale);
 p[0].y = (FPNum) this->giveNode(1)->giveUpdatedCoordinate(2,tStep,type,defScale);
 p[0].z = (FPNum) this->giveNode(1)->giveUpdatedCoordinate(3,tStep,type,defScale);
 p[1].x = (FPNum) this->giveNode(2)->giveUpdatedCoordinate(1,tStep,type,defScale);
 p[1].y = (FPNum) this->giveNode(2)->giveUpdatedCoordinate(2,tStep,type,defScale);
 p[1].z = (FPNum) this->giveNode(2)->giveUpdatedCoordinate(3,tStep,type,defScale);
 p[2].x = (FPNum) this->giveNode(3)->giveUpdatedCoordinate(1,tStep,type,defScale);
 p[2].y = (FPNum) this->giveNode(3)->giveUpdatedCoordinate(2,tStep,type,defScale);
 p[2].z = (FPNum) this->giveNode(3)->giveUpdatedCoordinate(3,tStep,type,defScale);
   
 go =  CreateTriangle3D(p);
 EGWithMaskChangeAttributes(WIDTH_MASK | COLOR_MASK | EDGE_COLOR_MASK | EDGE_FLAG_MASK | LAYER_MASK, go);
 EMAddGraphicsToModel(ESIModel(), go);
}


void
CCTPlate  :: drawScalar   (oofegGraphicContext& context)
{}

/*
void
CCTPlate  :: drawInternalState (oofegGraphicContext & gc)
//
// Draws internal state graphics representation
//
{
 WCRec p[3];
 GraphicObj *tr;
 double v1,v2,v3;
 DrawMode mode = gc.getDrawMode();
 TimeStep *tStep = domain->giveEngngModel()->giveCurrentStep();
 double defScale = gc.getDefScale();

 if (!gc.testElementGraphicActivity(this)) return; 
 
 // check for valid DrawMode
 if (!((mode == mxForce) || (mode == myForce) || (mode == mxyForce) ||
   (mode == szxForce) || (mode == syzForce))) return;
   
   

 EASValsSetLayer(OOFEG_STRESS_CONTOUR_LAYER);
 if (gc.getInternalVarsDefGeoFlag()) {
  // use deformed geometry
   p[0].x = (FPNum) this->giveNode(1)->giveUpdatedCoordinate(1,tStep,DisplacementVector,defScale);
   p[0].y = (FPNum) this->giveNode(1)->giveUpdatedCoordinate(2,tStep,DisplacementVector,defScale);
   p[0].z = (FPNum) this->giveNode(1)->giveUpdatedCoordinate(3,tStep,DisplacementVector,defScale);
   p[1].x = (FPNum) this->giveNode(2)->giveUpdatedCoordinate(1,tStep,DisplacementVector,defScale);
   p[1].y = (FPNum) this->giveNode(2)->giveUpdatedCoordinate(2,tStep,DisplacementVector,defScale);
   p[1].z = (FPNum) this->giveNode(2)->giveUpdatedCoordinate(3,tStep,DisplacementVector,defScale);
   p[2].x = (FPNum) this->giveNode(3)->giveUpdatedCoordinate(1,tStep,DisplacementVector,defScale);
   p[2].y = (FPNum) this->giveNode(3)->giveUpdatedCoordinate(2,tStep,DisplacementVector,defScale);
   p[2].z = (FPNum) this->giveNode(3)->giveUpdatedCoordinate(3,tStep,DisplacementVector,defScale);
  } else {
   p[0].x = (FPNum) this->giveNode(1)->giveCoordinate(1);
   p[0].y = (FPNum) this->giveNode(1)->giveCoordinate(2);
   p[0].z = (FPNum) this->giveNode(1)->giveCoordinate(3);
   p[1].x = (FPNum) this->giveNode(2)->giveCoordinate(1);
   p[1].y = (FPNum) this->giveNode(2)->giveCoordinate(2);
   p[1].z = (FPNum) this->giveNode(2)->giveCoordinate(3);
   p[2].x = (FPNum) this->giveNode(3)->giveCoordinate(1);
   p[2].y = (FPNum) this->giveNode(3)->giveCoordinate(2);
   p[2].z = (FPNum) this->giveNode(3)->giveCoordinate(3);
  }

 int result = 0;
 result+= this->giveInternalStateAtNode (gc, 1, &v1);
 result+= this->giveInternalStateAtNode (gc, 2, &v2);
 result+= this->giveInternalStateAtNode (gc, 3, &v3);

 if (result == 3) {
  tr = CreateTriangleWD3D (p,v1,v2,v3);
  EGWithMaskChangeAttributes(LAYER_MASK, tr);
  EMAddGraphicsToModel(ESIModel(), tr);
 }
}
*/


#endif
