/* $Header: /home/cvs/bp/oofem/sm/src/l4axisymm.C,v 1.3.4.1 2004/04/05 15:19:47 bp Exp $ */
/*

                   *****    *****   ******  ******  ***   ***                            
                 **   **  **   **  **      **      ** *** **                             
                **   **  **   **  ****    ****    **  *  **                              
               **   **  **   **  **      **      **     **                               
              **   **  **   **  **      **      **     **                                
              *****    *****   **      ******  **     **         
            
                                                                   
               OOFEM : Object Oriented Finite Element Code                 
                    
                 Copyright (C) 1993 - 2000   Borek Patzak                                       



         Czech Technical University, Faculty of Civil Engineering,
     Department of Structural Mechanics, 166 29 Prague, Czech Republic
                                                                               
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                                                                              
*/

//   file L4AXISYMM.CC
// 9.5.1995
//
 
#include "l4axisymm.h"
#include "node.h"
#include "material.h"
#include "gausspnt.h"
#include "gaussintegrationrule.h"
#include "flotmtrx.h"
#include "flotarry.h"
#include "intarray.h"
#include "domain.h"
#include "structuralcrosssection.h"
#include "structuralms.h"
#include "engngm.h"
#include "mathfem.h"
#ifndef __MAKEDEPEND
#include <stdio.h>
#endif

#ifdef __OOFEG
#include "oofeggraphiccontext.h"
#include "oofegutils.h"
#include "conTable.h"
#endif

FEI2dQuadLin L4Axisymm::interpolation (1,2);

L4Axisymm :: L4Axisymm (int n, Domain* aDomain)
         : NLStructuralElement (n,aDomain)
// Constructor.
{
 numberOfNodes  = 4;
}

L4Axisymm :: ~L4Axisymm ()
// destructor
{
}


Interface* 
L4Axisymm :: giveInterface (InterfaceType interface)
{
 if (interface == ZZNodalRecoveryModelInterfaceType) return (ZZNodalRecoveryModelInterface*) this;
 else if (interface == SPRNodalRecoveryModelInterfaceType) return (SPRNodalRecoveryModelInterface*) this;
 else if (interface == SpatialLocalizerInterfaceType) return (SpatialLocalizerInterface*) this;
 return NULL;
}



void
L4Axisymm :: computeNmatrixAt (GaussPoint* aGaussPoint, FloatMatrix& answer) 
     // Returns the displacement interpolation matrix {N} of the receiver,
     // evaluated at aGaussPoint.
{
  int i;
  FloatArray n (4);
  
  answer.resize(2,8);
  answer.zero();
  this->interpolation.evalN (n, *aGaussPoint->giveCoordinates(), 0.0); 
  
  for (i=1;i<=4;i++){
    answer.at(1,2*i-1) = n.at(i);
    answer.at(2,2*i-0) = n.at(i);
  }
  
  return ;
}


void
L4Axisymm :: computeBmatrixAt (GaussPoint *aGaussPoint, FloatMatrix& answer, int li, int ui)
//
// Returns the [6x8] strain-displacement matrix {B} of the receiver,
// evaluated at aGaussPoint.
// (epsilon_x,epsilon_y,...,Gamma_xy) = B . r
// r = ( u1,v1,u2,v2,u3,v3,u4,v4)
{
 int i;
 double r,x;
 int size, ind = 1;
 FloatMatrix  dnx ;

 this->interpolation.evaldNdx (dnx, this->giveDomain(), nodeArray, *aGaussPoint->giveCoordinates(), 0.0);

 if (ui == ALL_STRAINS) {size = 6; ui = 6;} else  size = ui-li+1;
 if ((size <0) || (size > 6)) _error ("ComputeBmatrixAt size mismatch");

 //gm = new FloatMatrix(size,8);
 answer.resize(size,8);
 answer.zero();

 if ((li <= 1) && (ui >= 1)) {
  for (i=1;i<=4;i++) answer.at(ind,2*i-1) = dnx.at(i,1);
  ind ++;
 }

 if ((li <=2) && (ui >= 2)) {
  for (i=1;i<=4;i++) answer.at(ind,2*i-0) = dnx.at(i,2);
  ind ++;
 }
 
 if ((li <=3) && (ui >= 3)) {
  FloatArray n (4);
  this->interpolation.evalN (n, *aGaussPoint->giveCoordinates(), 0.0); 

  r=0.;
  for (i=1;i<=numberOfNodes;i++) {
   x  = this -> giveNode (i) -> giveCoordinate (1);
   r += x*n.at(i);
  }

  answer.at(ind,1) = n.at(1)/r;
  answer.at(ind,3) = n.at(2)/r;
  answer.at(ind,5) = n.at(3)/r;
  answer.at(ind,7) = n.at(4)/r;

  ind++;
 }

 if ((li <=4) && (ui >= 4)) ind++;
 if ((li <=5) && (ui >= 5)) ind++;

 if ((li <=6) && (ui >= 6)) {
  for (i=1;i<=4;i++){
   answer.at(ind,2*i-1) = dnx.at(i,2);
   answer.at(ind,2*i-0) = dnx.at(i,1);
  }
 }

 return ;
}


void
L4Axisymm :: computeGaussPoints ()
// Sets up the array containing the four Gauss points of the receiver.
{
 numberOfIntegrationRules = 2 ;
 integrationRulesArray = new IntegrationRule*[2];
 integrationRulesArray[0] = new GaussIntegrationRule (1,domain, 1, 2);
  integrationRulesArray[0]->setUpIntegrationPoints (_Square, numberOfGaussPoints, this,  _3dMat);

  integrationRulesArray[1] = new GaussIntegrationRule (2,domain, 3, 6);
  integrationRulesArray[1]->setUpIntegrationPoints (_Square, numberOfFiAndShGaussPoints, this,  _3dMat);

}

double
L4Axisymm :: computeVolumeAround (GaussPoint* aGaussPoint)
// Returns the portion of the receiver which is attached to aGaussPoint.
{
 int          i;
 double       determinant,weight,volume,r,x;
 FloatArray n (4);
 
 this->interpolation.evalN (n, *aGaussPoint->giveCoordinates(), 0.0); 
 
 r=0.;
 for (i=1;i<=numberOfNodes;i++) {
  x  = this -> giveNode (i) -> giveCoordinate (1);
  r += x * n.at(i);
 }

 determinant = fabs(this->interpolation.giveTransformationJacobian (domain, nodeArray, 
                                   *aGaussPoint->giveCoordinates(), 0.0)); 

 weight      = aGaussPoint -> giveWeight ();
 volume      = determinant * weight * r;
 
 return volume;
}

IRResultType
L4Axisymm :: initializeFrom (InputRecord* ir)
{
 const char *__keyword, *__proc = "initializeFrom"; // Required by IR_GIVE_FIELD macro
 IRResultType result;                               // Required by IR_GIVE_FIELD macro

 this->NLStructuralElement :: initializeFrom (ir);
 
 numberOfGaussPoints          = 4;
 IR_GIVE_OPTIONAL_FIELD (ir, numberOfGaussPoints, "nip"); // Macro
 
 if (!((numberOfGaussPoints == 1) ||
   (numberOfGaussPoints == 4) ||
   (numberOfGaussPoints == 9) ||
   (numberOfGaussPoints == 16))) 
  numberOfGaussPoints = 4; 

 numberOfFiAndShGaussPoints = 1; 

 this -> computeGaussPoints();
 
 return IRRT_OK;
}



void
L4Axisymm :: computeStrainVector (FloatArray& answer, GaussPoint* gp, TimeStep* stepN)
// Computes the vector containing the strains at the Gauss point gp of
// the receiver, at time step stepN. The nature of these strains depends
// on the element's type.
{
 int i, rot;
 FloatMatrix b, A;
 FloatArray  u, Epsilon, help;
 fMode mode = domain->giveEngngModel()-> giveFormulation() ;
 
 answer.resize (6);
 answer.zero();
 rot     = this -> updateRotationMatrix () ;
 if (mode == TL) {     // Total Lagrange formulation
  this -> computeVectorOf(DisplacementVector,VM_Total,stepN, u) ;
  if (rot) u.rotatedWith(this->rotationMatrix, 'n') ;
   // linear part of strain tensor (in vector form)
  
  this -> computeBmatrixAt (gp,b,1,2);
  Epsilon.beProductOf (b, u);
  answer.at(1) = Epsilon.at(1);
  answer.at(2) = Epsilon.at(2);
  // delete Epsilon;  delete b;
  
  if (numberOfFiAndShGaussPoints == 1) {
   //
   // if reduced integration in one gp only
   // force the evaluation of eps_fi in this gauss point
   // instead of evaluating in given gp
   //
   GaussPoint *helpGaussPoint; 
   helpGaussPoint = integrationRulesArray[1]->getIntegrationPoint(0);
   
   this -> computeBmatrixAt (helpGaussPoint,b,3,6);
   
  } else {
   _error ("computeStrainVector: numberOfFiAndShGaussPoints size mismatch");
  }
  Epsilon.beProductOf (b, u);
  answer.at(3) = Epsilon.at(1);
  answer.at(6) = Epsilon.at(4);
  
  // delete Epsilon;  delete b;
  
  if (nlGeometry) {
   for (i = 1; i<= 6; i++) {
    // nonlin part of strain vector
    this -> computeNLBMatrixAt(A, gp, i);
    if (A.isNotEmpty()) {
     help.beProductOf(A, u);
     answer.at(i) += 0.5 * dotProduct(u, help ,u.giveSize());
     // delete help;
     //delete A;
    }
   }
  }
  // delete u;
  //
 } else if (mode == AL) {  // actualized Lagrange formulation
  _error ("computeStrainVector : unsupported mode");
 }
 
 return ;
}


#define NONZERO_COORD_TOL 1.e-2

double
L4Axisymm :: giveCharacteristicLenght (GaussPoint* gp, const FloatArray& normalToCrackPlane)
//
// returns receivers characteristic length in gp (for some material models)
// for crack formed in plane with normal normalToCrackPlane.
//
{
  // return this -> giveLenghtInDir(normalToCrackPlane) / sqrt (this->numberOfGaussPoints);
 //
 // we must handle special case - crack caused by hoop stres Sigma_z
 // note: z-axis is always principal axis, because there exist only one
 // nonzero shear strain (Sigma_xy), which will cause only rotation of
 // principal axises in x-y plane (around z-axis).
 // so this yields following : 
 // normalCrackPlane(3) component  can be only zero
 // (then crack normal lies in x-y plane) or equal to 1.0 (crack normal 
 // is perpendicular to to x--y plane) - crack caused by hoop strain.
 if (fabs (normalToCrackPlane.at(3)) > NONZERO_COORD_TOL) {
  double r = 0.;
  int i;
  for (i=1; i<= numberOfNodes ; i++) 
   r += this->giveNode(i)->giveCoordinate (1);
  r = r / ((double)numberOfNodes);
  return r;
 } else {
  return this -> giveLenghtInDir(normalToCrackPlane);
 }
}

void
L4Axisymm ::   giveNodeDofIDMask  (int inode, IntArray& answer) const {
// returns DofId mask array for inode element node.
// DofId mask array determines the dof ordering requsted from node.
// DofId mask array contains the DofID constants (defined in cltypes.h)
// describing physical meaning of particular DOFs.
 //IntArray* answer = new IntArray (2);
 answer.resize (2);

 answer.at(1) = D_u;
 answer.at(2) = D_v;

 return ;
}

int
L4Axisymm :: computeGlobalCoordinates (FloatArray& answer, const FloatArray& lcoords) 
{
 this->interpolation.local2global (answer, domain, nodeArray, lcoords, 0.0); 
 return 1;

}

int
L4Axisymm::computeLocalCoordinates (FloatArray& answer, const FloatArray& coords)
{
 return this->interpolation.global2local (answer, domain, nodeArray, coords, 0.0); 
}

void 
L4Axisymm::SPRNodalRecoveryMI_giveSPRAssemblyPoints (IntArray& pap)
{
 pap.resize(4);
 for (int i=1; i <5; i++) pap.at(i) = this->giveNode(i)->giveNumber();
}

void
L4Axisymm::SPRNodalRecoveryMI_giveDofMansDeterminedByPatch (IntArray& answer, int pap)
{
 int found = 0;
 answer.resize(1);

 for (int i=1; i<5; i++) if (pap == this->giveNode(i)->giveNumber()) found = 1;
 if (found) answer.at(1) = pap;
 else _error ("SPRNodalRecoveryMI_giveDofMansDeterminedByPatch: node unknown");
}

int
L4Axisymm::SPRNodalRecoveryMI_giveNumberOfIP ()
{ return this->giveDefaultIntegrationRulePtr()->getNumberOfIntegrationPoints(); }


void
L4Axisymm::SPRNodalRecoveryMI_computeIPGlobalCoordinates (FloatArray& coords, GaussPoint* gp)
{
 this->computeGlobalCoordinates(coords, *gp->giveCoordinates());
}

SPRPatchType
L4Axisymm::SPRNodalRecoveryMI_givePatchType()
{
 return SPRPatchType_2dxy;
}

int 
L4Axisymm::SpatialLocalizerI_containsPoint (const FloatArray& coords) {

 int result;
 FloatArray lcoords;
 result = this->computeLocalCoordinates (lcoords, coords);

 return result;
}


int 
L4Axisymm :: ZZNodalRecoveryMI_giveDofManRecordSize(InternalStateType type)
{
 if ((type == IST_StressTensor)||(type ==IST_StrainTensor)) return 6;
 
 GaussPoint *gp = integrationRulesArray[0]-> getIntegrationPoint(0) ;
 return this->giveIPValueSize (type, gp);
}

void
L4Axisymm :: ZZNodalRecoveryMI_ComputeEstimatedInterpolationMtrx  
(FloatMatrix& answer, GaussPoint* aGaussPoint, InternalStateType type)
{
  // evaluates N matrix (interpolation estimated stress matrix)
  // according to Zienkiewicz & Zhu paper
  // N(nsigma, nsigma*nnodes)
  // Definition : sigmaVector = N * nodalSigmaVector
 FloatArray n;
 this->interpolation.evalN (n, *aGaussPoint->giveCoordinates(), 0.0); 

 if (this->giveIPValueSize(type, aGaussPoint)) answer.resize(1,4) ;
 else return;

 answer.zero();
 
 answer.at(1,1) = n.at(1) ;
 answer.at(1,2) = n.at(2) ;
 answer.at(1,3) = n.at(3) ;
 answer.at(1,4) = n.at(4) ;

   return ;
}


void
L4Axisymm :: computeEgdeNMatrixAt (FloatMatrix& answer, GaussPoint* aGaussPoint) 
{
  /*
  
  computes interpolation matrix for element edge.
  we assemble locally this matrix for only nonzero 
  shape functions. 
  (for example only two nonzero shape functions for 2 dofs are
  necessary for linear plane stress tringle edge).
  These nonzero shape functions are then mapped to 
  global element functions.
  
  Using mapping technique will allow to assemble shape functions 
  without regarding particular side
  */
 
  FloatArray n(2);
  this->interpolation.edgeEvalN (n, *aGaussPoint->giveCoordinates(), 0.0); 
  
  answer.resize (2,4); answer.zero();
  
  answer.at(1,1) = n.at(1) ;
  answer.at(1,3) = n.at(2) ;
  answer.at(2,2) = n.at(1) ;
  answer.at(2,4) = n.at(2) ;
}


void
L4Axisymm :: giveEdgeDofMapping (IntArray& answer, int iEdge) const
{
  /* 
   provides dof mapping of local edge dofs (only nonzero are taken into account)
   to global element dofs
  */

  answer.resize(4);
  if (iEdge == 1)  { // edge between nodes 1,2
  answer.at(1) = 1;
  answer.at(2) = 2;
  answer.at(3) = 3;
  answer.at(4) = 4;
  } else if (iEdge == 2) { // edge between nodes 2 3
  answer.at(1) = 3;
  answer.at(2) = 4;
  answer.at(3) = 5;
  answer.at(4) = 6;
  } else if (iEdge == 3) {// edge between nodes 3 4
  answer.at(1) = 5;
  answer.at(2) = 6;
  answer.at(3) = 7;
  answer.at(4) = 8;
  } else if (iEdge == 4) {// edge between nodes 4 1
  answer.at(1) = 7;
  answer.at(2) = 8;
  answer.at(3) = 1;
  answer.at(4) = 2;
  } else {
  _error ("giveEdgeDofMapping: wrong edge number");
  }

  return ;
}

double
L4Axisymm ::   computeEdgeVolumeAround (GaussPoint* aGaussPoint, int iEdge) 
{
  FloatArray c(2);
  this->computeEdgeIpGlobalCoords (c, aGaussPoint, iEdge) ;
  double result = this->interpolation.edgeGiveTransformationJacobian (iEdge, domain, nodeArray, 
                                                                      *aGaussPoint->giveCoordinates(), 0.0); 
  
  
  return c.at(1)* result * aGaussPoint -> giveWeight() ;
}

void      
L4Axisymm::computeEdgeIpGlobalCoords (FloatArray& answer, GaussPoint* gp, int iEdge) 
{
 this->interpolation.edgeLocal2global (answer, iEdge, domain, nodeArray, *gp->giveCoordinates(), 0.0); 
}


int
L4Axisymm :: computeLoadLEToLRotationMatrix (FloatMatrix& answer, int iEdge, GaussPoint* gp) 
{
  // returns transformation matrix from
  // edge local coordinate system
  // to element local c.s 
  // (same as global c.s in this case) 
  //
  // i.e. f(element local) = T * f(edge local)
  //
  double dx,dy, length ;
  Node   *nodeA,*nodeB ;
 IntArray edgeNodes(2);

 answer.resize(2,2);
 answer.zero();

 this->interpolation.computeEdgeMapping (edgeNodes, nodeArray, iEdge); 

 // edge nodes are global numbers, not local element numbers
  nodeA   = domain->giveNode(edgeNodes.at(1)) ;
  nodeB   = domain->giveNode(edgeNodes.at(2)) ;

  dx      = nodeB->giveCoordinate(1) - nodeA->giveCoordinate(1) ;
  dy      = nodeB->giveCoordinate(2) - nodeA->giveCoordinate(2) ;
  length = sqrt(dx*dx + dy*dy);

  answer.at(1,1) = dx/length;
  answer.at(1,2) = -dy/length;
  answer.at(2,1) = answer.at(1,2);
  answer.at(2,2) = dx/length;

  return 1;
}




#ifdef __OOFEG
//#include "rcm2.h"
#define TR_LENGHT_REDUCT 0.3333

void L4Axisymm :: drawRawGeometry (oofegGraphicContext &gc)
{
  WCRec p[4];
  GraphicObj *go;

 if (!gc.testElementGraphicActivity(this)) return; 

  EASValsSetLineWidth(OOFEG_RAW_GEOMETRY_WIDTH);
  EASValsSetColor(gc.getElementColor());
  EASValsSetEdgeColor(gc.getElementEdgeColor());
  EASValsSetEdgeFlag(TRUE);
  EASValsSetLayer(OOFEG_RAW_GEOMETRY_LAYER);
 EASValsSetFillStyle (FILL_HOLLOW);
  p[0].x = (FPNum) this->giveNode(1)->giveCoordinate(1);
  p[0].y = (FPNum) this->giveNode(1)->giveCoordinate(2);
  p[0].z = 0.;
  p[1].x = (FPNum) this->giveNode(2)->giveCoordinate(1);
  p[1].y = (FPNum) this->giveNode(2)->giveCoordinate(2);
  p[1].z = 0.;
  p[2].x = (FPNum) this->giveNode(3)->giveCoordinate(1);
  p[2].y = (FPNum) this->giveNode(3)->giveCoordinate(2);
  p[2].z = 0.;
  p[3].x = (FPNum) this->giveNode(4)->giveCoordinate(1);
  p[3].y = (FPNum) this->giveNode(4)->giveCoordinate(2);
  p[3].z = 0.;
   
  go =  CreateQuad3D(p);
  EGWithMaskChangeAttributes(WIDTH_MASK | FILL_MASK | COLOR_MASK | EDGE_COLOR_MASK | EDGE_FLAG_MASK | LAYER_MASK, go);
  EGAttachObject(go, (EObjectP) this);
  EMAddGraphicsToModel(ESIModel(), go);
}


void L4Axisymm :: drawDeformedGeometry (oofegGraphicContext &gc, UnknownType type)
{
  WCRec p[4];
  GraphicObj *go;
  TimeStep *tStep = domain->giveEngngModel()->giveCurrentStep();
 double defScale = gc.getDefScale();

 if (!gc.testElementGraphicActivity(this)) return; 

  EASValsSetLineWidth(OOFEG_DEFORMED_GEOMETRY_WIDTH);
  EASValsSetColor(gc.getDeformedElementColor());
  EASValsSetEdgeColor(gc.getElementEdgeColor());
  EASValsSetEdgeFlag(TRUE);
  EASValsSetLayer(OOFEG_DEFORMED_GEOMETRY_LAYER);
 EASValsSetFillStyle (FILL_HOLLOW);
  p[0].x = (FPNum) this->giveNode(1)->giveUpdatedCoordinate(1,tStep,type,defScale);
  p[0].y = (FPNum) this->giveNode(1)->giveUpdatedCoordinate(2,tStep,type,defScale);
  p[0].z = 0.;
  p[1].x = (FPNum) this->giveNode(2)->giveUpdatedCoordinate(1,tStep,type,defScale);
  p[1].y = (FPNum) this->giveNode(2)->giveUpdatedCoordinate(2,tStep,type,defScale);
  p[1].z = 0.;
  p[2].x = (FPNum) this->giveNode(3)->giveUpdatedCoordinate(1,tStep,type,defScale);
  p[2].y = (FPNum) this->giveNode(3)->giveUpdatedCoordinate(2,tStep,type,defScale);
  p[2].z = 0.;
  p[3].x = (FPNum) this->giveNode(4)->giveUpdatedCoordinate(1,tStep,type,defScale);
  p[3].y = (FPNum) this->giveNode(4)->giveUpdatedCoordinate(2,tStep,type,defScale);
  p[3].z = 0.;
   
  go =  CreateQuad3D(p);
  EGWithMaskChangeAttributes(WIDTH_MASK | FILL_MASK | COLOR_MASK | EDGE_COLOR_MASK | EDGE_FLAG_MASK | LAYER_MASK, go);
  EMAddGraphicsToModel(ESIModel(), go);
}



void L4Axisymm :: drawScalar   (oofegGraphicContext& context)
{
 int i, indx, result = 0;
  WCRec p[4];
  GraphicObj *tr;
 TimeStep* tStep = this->giveDomain()->giveEngngModel()->giveCurrentStep();
 FloatArray v[4];
 double s[4], defScale;
 IntArray map;

 if (!context.testElementGraphicActivity(this)) return; 

 EASValsSetLayer(OOFEG_VARPLOT_PATTERN_LAYER);
 if (context.giveIntVarMode() == ISM_recovered) {
  for (i=1; i<=4; i++) 
   result+= this->giveInternalStateAtNode (v[i-1], context.giveIntVarType(), context.giveIntVarMode(), i, tStep);
  if (result !=4) return;

  this->giveIntVarCompFullIndx (map, context.giveIntVarType());
  if ((indx = map.at(context.giveIntVarIndx())) == 0) return;
  for (i=1; i<=4; i++) s[i-1]=v[i-1].at(indx);
  
  if (context.getScalarAlgo() == SA_ISO_SURF) {
   
   for (i=0; i< 4; i++) {
    if (context.getInternalVarsDefGeoFlag()) {
     // use deformed geometry
     defScale = context.getDefScale();
     p[i].x = (FPNum) this->giveNode(i+1)->giveUpdatedCoordinate(1,tStep,DisplacementVector,defScale);
     p[i].y = (FPNum) this->giveNode(i+1)->giveUpdatedCoordinate(2,tStep,DisplacementVector,defScale);
     p[i].z = 0.;
     
    } else {
     p[i].x = (FPNum) this->giveNode(i+1)->giveCoordinate(1);
     p[i].y = (FPNum) this->giveNode(i+1)->giveCoordinate(2);
     p[i].z = 0.;
    }
   }
   
   //EASValsSetColor(gc.getYieldPlotColor(ratio));
   context.updateFringeTableMinMax (s, 4);
   tr =  CreateQuadWD3D(p, s[0], s[1], s[2], s[3]);
   EGWithMaskChangeAttributes(LAYER_MASK, tr);
   EMAddGraphicsToModel(ESIModel(), tr);
   
   /*
   } else if (context.getScalarAlgo() == SA_ISO_LINE) {
   
   EASValsSetColor(context.getActiveCrackColor());
   EASValsSetLineWidth(OOFEG_ISO_LINE_WIDTH);
   
   for (i=0; i< 4; i++) {
    if (context.getInternalVarsDefGeoFlag()) {
     // use deformed geometry
     defScale = context.getDefScale();
     p[i].x = (FPNum) this->giveNode(i+1)->giveUpdatedCoordinate(1,tStep,DisplacementVector,defScale);
     p[i].y = (FPNum) this->giveNode(i+1)->giveUpdatedCoordinate(2,tStep,DisplacementVector,defScale);
     p[i].z = 0.;
     
    } else {
     p[i].x = (FPNum) this->giveNode(i+1)->giveCoordinate(1);
     p[i].y = (FPNum) this->giveNode(i+1)->giveCoordinate(2);
     p[i].z = 0.;
    }
   }
   
   // isoline implementation
   oofeg_drawIsoLinesOnQuad (p, s);

   */
  } 
 } else if (context.giveIntVarMode() == ISM_local) {
  
  if (numberOfGaussPoints != 4) return;
  int ip;
  GaussPoint* gp;
  IntArray ind(4);
  FloatArray* gpCoords;
  WCRec pp[9];
  
  for (i=0; i< 4; i++) {
   if (context.getInternalVarsDefGeoFlag()) {
    // use deformed geometry
    defScale = context.getDefScale();
    pp[i].x = (FPNum) this->giveNode(i+1)->giveUpdatedCoordinate(1,tStep,DisplacementVector,defScale);
    pp[i].y = (FPNum) this->giveNode(i+1)->giveUpdatedCoordinate(2,tStep,DisplacementVector,defScale);
    pp[i].z = 0.;
    
   } else {
    pp[i].x = (FPNum) this->giveNode(i+1)->giveCoordinate(1);
    pp[i].y = (FPNum) this->giveNode(i+1)->giveCoordinate(2);
    pp[i].z = 0.;
   }
  }
  
  for (i=0; i<3; i++) {
   pp[i+4].x = 0.5*(pp[i].x+pp[i+1].x); pp[i+4].y = 0.5*(pp[i].y+pp[i+1].y); pp[i+4].z = 0.5*(pp[i].z+pp[i+1].z);
  }
  pp[7].x = 0.5*(pp[3].x+pp[0].x); pp[7].y = 0.5*(pp[3].y+pp[0].y); pp[7].z = 0.5*(pp[3].z+pp[0].z);
   
  pp[8].x = 0.25*(pp[0].x+pp[1].x+pp[2].x+pp[3].x); 
  pp[8].y = 0.25*(pp[0].y+pp[1].y+pp[2].y+pp[3].y);
  pp[8].z = 0.25*(pp[0].z+pp[1].z+pp[2].z+pp[3].z);
  
  for (ip=1 ; ip<= numberOfGaussPoints ; ip++) {
   gp = integrationRulesArray[0]-> getIntegrationPoint(ip-1) ;
   gpCoords = gp->giveCoordinates();
   if ((gpCoords->at(1) >0.)&&(gpCoords->at(2) >0.)) {
    ind.at(1) = 0; ind.at(2) = 4; ind.at(3) = 8; ind.at(4) = 7;
   } else if ((gpCoords->at(1) < 0.)&&(gpCoords->at(2) >0.)) {
    ind.at(1) = 4; ind.at(2) = 1; ind.at(3) = 5; ind.at(4) = 8;
   } else if ((gpCoords->at(1) < 0.)&&(gpCoords->at(2) <0.)) {
    ind.at(1) = 5; ind.at(2) = 2; ind.at(3) = 6; ind.at(4) = 8;
   } else {
    ind.at(1) = 6; ind.at(2) = 3; ind.at(3) = 7; ind.at(4) = 8;
   }
   
   if (giveIPValue (v[0], gp, context.giveIntVarType(), tStep) == 0 ) return;
   this->giveIntVarCompFullIndx (map, context.giveIntVarType());
   if ((indx = map.at(context.giveIntVarIndx())) == 0) return;
   for (i=1; i<=4; i++) s[i-1]=v[0].at(indx);
   
   for (i=0; i<4; i++) {
    p[i].x = pp[ind.at(i+1)].x;
    p[i].y = pp[ind.at(i+1)].y;
    p[i].z = pp[ind.at(i+1)].z;
   }
   context.updateFringeTableMinMax (s, 4);   
   tr =  CreateQuadWD3D(p, s[0], s[1], s[2], s[3]);
   EGWithMaskChangeAttributes(LAYER_MASK, tr);
   EMAddGraphicsToModel(ESIModel(), tr);
  }
 }
 return;
}


#endif
