#cell types
#linear triangle
Triangle_2d_1=1
Quad_2d_1=2
Tetrahedron_3d_1=3

class Cell:
    """
    Representation of computational cell. It is an attribute of a domain.
    
    It is asumed, that cell can return its vertices, they can be either attributes of a cell
    (unstructured grids) or verices can be generated based on cell number and domain (structured grids). 
    In the latter case, the derived class is to be created to handle this correctly.
    
    """
    def __init__(self, mesh, number, label):
        self.mesh=mesh
        self.number=number
        self.label=label

    def __init__(self, mesh, number, label, vertices):
        self.mesh=mesh
        self.number=number
        self.label=label
        self.vertices=vertices

    def giveVertices (self):
        """Returns the list of cell vertices"""
        return self.vertices

    def containsPoint (self, point):
        """Returns true if cell contains given point"""
    
    def interpolate (self, point, vertexValues):
        """Interpolates given vertex values to given point)"""
    
    def giveGeometryType(self):
        """Returns geometry type of receiver"""



class Triangle_2d_lin(Cell):
    """Unstructured 2d triangular element with linear interpolation"""

    def giveGeometryType(self):
        """Returns geometry type of receiver"""
        return Triangle_2d_1

    def glob2loc(self, coords):
        """Converts global coordinates to local (area) coordinates"""
        c1=mesh.giveVertex(self.vertices[0]).giveCoordinates()
        c2=mesh.giveVertex(self.vertices[1]).giveCoordinates()
        c3=mesh.giveVertex(self.vertices[2]).giveCoordinates()
        x1 = c1.coords[0]; y1 = c1.coords[1]
        x2 = c2.coords[0]; y2 = c2.coords[1]
        x3 = c3.coords[0]; y3 = c3.coords[1]
        
        area = 0.5 * ( x2 * y3 + x1 * y2 + y1 * x3 - x2 * y1 - x3 * y2 - x1 * y3 );
        
        l1 = ( ( x2 * y3 - x3 * y2 ) + ( y2 - y3 ) * coords[0] + ( x3 - x2 ) * coords[1] ) / 2. / area;
        l2 = ( ( x3 * y1 - x1 * y3 ) + ( y3 - y1 ) * coords[0] + ( x1 - x3 ) * coords[1] ) / 2. / area;
        l3 = ( ( x1 * y2 - x2 * y1 ) + ( y1 - y2 ) * coords[0] + ( x2 - x1 ) * coords[1] ) / 2. / area;

        return (l1,l2,l3)

        
    def interpolate(self, point, vertexValues):
        
        ac = self.glob2loc(point)
        return vertexValues[0]*ac[0]+vertexValues[1]*ac[1]+vertexValues[2]*ac[2]

    def containsPoint(self, point):
        ac = self.glob2loc(point)
        
        for li in ac:
            if li < -0.001 or li > 1.001:
                return False
        return True;

    
class Quad_2d_lin(Cell):
    """Unstructured 2d quad element with linear interpolation"""

    def giveGeometryType(self):
        """Returns geometry type of receiver"""
        return Quad_2d_1

    def glob2loc(self, coords):
        """Converts global coordinates to local (area) coordinates"""
        return (0.0,0.0)

        
    def interpolate(self, point, vertexValues):
        
        ac = self.glob2loc(point)
        return (0.25*(1.0+ac[0])*(1.0+ac[1])*vertexValues[0]+
                0.25*(1.0-ac[0])*(1.0+ac[1])*vertexValues[1]+
                0.25*(1.0-ac[0])*(1.0-ac[1])*vertexValues[2]+
                0.25*(1.0+ac[0])*(1.0-ac[1])*vertexValues[3])


    def containsPoint(self, point):
        ac = self.glob2loc(point)
        
        for li in ac:
            if li < -1.001 or li > 1.001:
                return False
        return True;

class Tetrahedron_3d_lin(Cell):
    """Unstructured 3d tetrahedral element with linear interpolation"""

    def giveGeometryType(self):
        """Returns geometry type of receiver"""
        return Tetrahedron_3d_1

    def glob2loc(self, coords):
        """Converts global coordinates to local (area) coordinates"""
        c1=mesh.giveVertex(self.vertices[0]).giveCoordinates()
        c2=mesh.giveVertex(self.vertices[1]).giveCoordinates()
        c3=mesh.giveVertex(self.vertices[2]).giveCoordinates()
        c4=mesh.giveVertex(self.vertices[3]).giveCoordinates()
        
        x1 = c1.coords[0]; y1 = c1.coords[1]
        x2 = c2.coords[0]; y2 = c2.coords[1]
        x3 = c3.coords[0]; y3 = c3.coords[1]
        x4 = c4.coords[0]; y4 = c4.coords[1]
        
        volume = ( ( x4 - x1 ) * ( y2 - y1 ) * ( z3 - z1 ) - ( x4 - x1 ) * ( y3 - y1 ) * ( z2 - z1 ) +
                   ( x3 - x1 ) * ( y4 - y1 ) * ( z2 - z1 ) - ( x2 - x1 ) * ( y4 - y1 ) * ( z3 - z1 ) +
                   ( x2 - x1 ) * ( y3 - y1 ) * ( z4 - z1 ) - ( x3 - x1 ) * ( y2 - y1 ) * ( z4 - z1 ) ) / 6.;

        
        l1 = ( ( x3 - x2 ) * ( yp - y2 ) * ( z4 - z2 ) - ( xp - x2 ) * ( y3 - y2 ) * ( z4 - z2 ) +
               ( x4 - x2 ) * ( y3 - y2 ) * ( zp - z2 ) - ( x4 - x2 ) * ( yp - y2 ) * ( z3 - z2 ) +
               ( xp - x2 ) * ( y4 - y2 ) * ( z3 - z2 ) - ( x3 - x2 ) * ( y4 - y2 ) * ( zp - z2 ) ) / 6. / volume;

        l2 = ( ( x4 - x1 ) * ( yp - y1 ) * ( z3 - z1 ) - ( xp - x1 ) * ( y4 - y1 ) * ( z3 - z1 ) +
               ( x3 - x1 ) * ( y4 - y1 ) * ( zp - z1 ) - ( x3 - x1 ) * ( yp - y1 ) * ( z4 - z1 ) +
               ( xp - x1 ) * ( y3 - y1 ) * ( z4 - z1 ) - ( x4 - x1 ) * ( y3 - y1 ) * ( zp - z1 ) ) / 6. / volume;
        
        l3 = ( ( x2 - x1 ) * ( yp - y1 ) * ( z4 - z1 ) - ( xp - x1 ) * ( y2 - y1 ) * ( z4 - z1 ) +
               ( x4 - x1 ) * ( y2 - y1 ) * ( zp - z1 ) - ( x4 - x1 ) * ( yp - y1 ) * ( z2 - z1 ) +
               ( xp - x1 ) * ( y4 - y1 ) * ( z2 - z1 ) - ( x2 - x1 ) * ( y4 - y1 ) * ( zp - z1 ) ) / 6. / volume;
        
        l4  = 1.0 - answer.at(1) - answer.at(2) - answer.at(3);

        return (l1,l2,l3,l4)

        
    def interpolate(self, point, vertexValues):
        
        ac = self.glob2loc(point)
        return vertexValues[0]*ac[0]+vertexValues[1]*ac[1]+vertexValues[2]*ac[2]+vertexValues[3]*ac[3]

    def containsPoint(self, point):
        ac = self.glob2loc(point)
        
        for li in ac:
            if li < -0.001 or li > 1.001:
                return False
        return True;
