class ViewContext:
    """
    Determines the mapping between master domain and associated view.
    Defines the local numbering of mapped vertices and cells, 
    provides mapping method between local component number (vertex or cell number)
    and corresponding global number.
    This class is supposed to be an abstract class, derived classes are assumed to define
    mapping.

    To be more general, the mapping should be also capable to map also element edges and surfaces, as well to adress
    components on remote/distrubuted nodes. This info can be returned as a tuple, containing all information.
    At present DataRecord is assumed to have following format:
    (number,)

    """

    def giveNumberOfVertices(self):
        """Returns the number of local (mapped) vertices"""
    def giveNumberOfCells(self):
        """Returns the number of local (mapped) cells"""
    def giveMasterVertexData(self, localVertexID):
        """Returns the master vertex info (including number, etc) corresponding to given local number (id)"""
    def giveMasterCellData(self, localCellID):
        """Returns the master cell info (number, etc.) corresponding to given local number (id)"""
    def giveMasterVertexNumber(self, localVertexID):
        """Returns the master vertex number corresponding to given local number (id)"""
    def giveMasterCellNumber(self, localCellID):
        """Returns the master cell number corresponding to given local number (id)"""

class ExplicitViewContext(ViewContext):
    """This view context is defined by (explicitly given) dictionary."""
    def __init__(self, vertexDict, cellDict):
        self.__vertexMap = vertexDict
        self.__cellMap = cellDict
        self.__numberOfVertices = len(self.__vertexMap)
        self.__numberOfCells    = len(self.__cellMap)

    def giveNumberOfVertices(self):
        """Returns the number of local (mapped) vertices"""
        return self.__numberOfVertices

    def giveNumberOfCells(self):
        """Returns the number of local (mapped) cells"""
        return self.__numberOfCells

    def giveMasterVertexNumber(self, localVertexID):
        """Returns the master vertex number corresponding to given local number (id)"""
        return self.__vertexMap[localVertexID]

    def giveMasterCellNumberData(self, localCellID):
        """Returns the master cell number corresponding to given local number (id)"""
        return self.__cellMap[localCellID]
        

