#
#               MuPIF: Multi-Physics Integration Framework 
#                   Copyright (C) 2010 Borek Patzak
#
#       Czech Technical University, Faculty of Civil Engineering,
#       Department of Mechanics, 166 29 Prague, Czech Republic
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

class ViewContext:
    """
    Determines the mapping between master domain and associated view.
    Defines the local numbering of mapped vertices and cells, 
    provides mapping method between local component number (vertex or cell number)
    and corresponding global number.
    This class is supposed to be an abstract class, derived classes are assumed to define
    mapping.

    To be more general, the mapping should be also capable to map also element edges and surfaces, as well to adress
    components on remote/distrubuted nodes. This info can be returned as a tuple, containing all information.
    At present DataRecord is assumed to have following format:
    (number,)

    """

    def giveNumberOfVertices(self):
        """Returns the number of local (mapped) vertices"""
    def giveNumberOfCells(self):
        """Returns the number of local (mapped) cells"""
    def giveMasterVertexData(self, localVertexID):
        """Returns the master vertex info (including number, etc) corresponding to given local number (id)"""
    def giveMasterCellData(self, localCellID):
        """Returns the master cell info (number, etc.) corresponding to given local number (id)"""
    def giveMasterVertexNumber(self, localVertexID):
        """Returns the master vertex number corresponding to given local number (id)"""
    def giveMasterCellNumber(self, localCellID):
        """Returns the master cell number corresponding to given local number (id)"""
    def merge(self, vc):
        """Merges the receiver and given context into the receiver"""

class ExplicitViewContext(ViewContext):
    """This view context is defined by (explicitly given) dictionary."""
    def __init__(self, vertexDict, cellDict):
        self.vertexMap = vertexDict
        self.cellMap = cellDict
        self.numberOfVertices = len(self.vertexMap)
        self.numberOfCells    = len(self.cellMap)

    def __init__(self):
        self.vertexMap = []
        self.cellMap = []
        self.numberOfVertices = 0
        self.numberOfCells    = 0
        

    def giveNumberOfVertices(self):
        """Returns the number of local (mapped) vertices"""
        return self.numberOfVertices

    def giveNumberOfCells(self):
        """Returns the number of local (mapped) cells"""
        return self.numberOfCells

    def giveMasterVertexNumber(self, localVertexID):
        """Returns the master vertex number corresponding to given local number (id)"""
        return self.vertexMap[localVertexID]

    def giveMasterCellNumberData(self, localCellID):
        """Returns the master cell number corresponding to given local number (id)"""
        return self.cellMap[localCellID]

    def merge(self, vc):
        """Merges the receiver and given context into the receiver"""
        if (type(vc) != type(ExplicitViewContext)):
            raise TypeError ("type mismatch when merging receiver and parameter")
        # merge together, check for duplicated entries (based on target entity numbers)
        # build vertex dictionary (key is target number)
        vertexDict={} # key is target number, value is local number
        mergedVertexMap = []
        for v in self.vertexMap:
            if (vertexDict.has_key(v)):
                # entry already exist
                pass
            else:
                vertexDict[v]=True
                mergedVertexMap.append(v)
        #loop over vc context and merge entries
        for v in vc.vertexMap:
            if (vertexDict.has_key(v)):
                # entry already exist
                pass
            else:
                vertexDict[v]=True
                mergedVertexMap.append(v)
        self.vertexMap = mergedVertexMap
        
        #merge cell maps
        cellDict={} # key is target number, value is local number
        mergedCellMap = []
        for v in self.cellMap:
            if (cellDict.has_key(v)):
                # entry already exist
                pass
            else:
                cellDict[v]=True
                mergedCellMap.append(v)
        #loop over vc context and merge entries
        for v in vc.cellMap:
            if (cellDict.has_key(v)):
                # entry already exist
                pass
            else:
                cellDict[v]=True
                mergedCellMap.append(v)
        self.cellMap = mergedCellMap
        
        


        
import bbox
import copy
import time

class BBoxViewContext(ExplicitViewContext):
    """This view context is defined by given BBox. It will contain all elements hit by given BBox and corresponding nodes.""" 
    def __init__(self, box, masterMesh, mirrorMesh=None):
        ExplicitViewContext.__init__(self)
        # request a list of mastres cells within a given box
        cells = masterMesh.giveCellLocalizer().giveItemsInBBox(box)
        # create an emtpy set for cell vertices
        vertices = set()
        # loop over master cells
        for icell in cells:
            # add icell into cell map
            self.cellMap.append((icell.number,))
            # get list of icell vertices
            icellvertices = icell.giveVertices()
            vertices.update(icellvertices)
        # now we have cell and vertices subset ready
        # cells are in a list, so already with local numbering
        # do local vertex renumbering
        for ivertex in vertices:
            self.vertexMap.append((ivertex,))

        self.numberOfVertices = len(vertices)
        self.numberOfCells = len(cells)

        if mirrorMesh:
            vertexGtoLmap = {}
            num = 0;
            # assemble global->local map first
            for ivertex in vertices:
                vertexGtoLmap[ivertex]=num
                num+=1
            # setup mirror Mesh data
            # build local vertex list
            vertexList=[]   
            num = 0; 
            for ivertex in vertices:
                #create a copy of master vertex
                v = copy.deepcopy(masterMesh.giveVertex(ivertex))
                v.number=num
                num+=1
                vertexList.append(v)

            # build local cell list
            cellList=[]
            num=0
            for icell in cells:
                #create a copy of master cell
                #c = copy.deepcopy(icell) # this is not a good choice (cells contain domain link!)
                c = icell.copy() # shalow copy here

                c.number=num
                num+=1
                #renumber vertices to local numbering
                icellvertices = [ vertexGtoLmap[i] for i in icell.giveVertices()]
                c.vertices = icellvertices
                c.mesh = mirrorMesh
                cellList.append(c)
            # setup mirrored mesh
            mirrorMesh.setup (vertexList, cellList)
            mirrorMesh.mapping = self

                
               
            

                

        
        
