#
#               MuPIF: Multi-Physics Integration Framework 
#                   Copyright (C) 2010 Borek Patzak
#
#       Czech Technical University, Faculty of Civil Engineering,
#       Department of Mechanics, 166 29 Prague, Czech Republic
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#


import math
import field
import mesh
import cell
import timestep
import re
from numpy import array, arange, random, zeros
from tvtk.api import tvtk
from mayavi.sources.vtk_data_source import VTKDataSource
from mayavi.core.dataset_manager import DatasetManager 

debug = False

def create_tvtk_ug(mesh):
    """generate tvtk.UnstructuredGrid grid with different cell types.
    Returns a created unstructured grid
    """
    points=[]
    for vertex in mesh.vertices():
        points.append(vertex.coords)
    cells=[]
    cellTypes=[]
    offsets=[]
    for _cell in mesh.cells():
        #print "huu "
        offsets.append(len(cells))
        cells.append(len(_cell.vertices))
        cells.extend(_cell.vertices)
        if (_cell.giveGeometryType() == cell.Quad_2d_1):
            cellTypes.append(tvtk.Quad().cell_type)
        elif (_cell.giveGeometryType() == cell.Triangle_2d_1):
            cellTypes.append(tvtk.Triangle().cell_type)
        elif (_cell.giveGeometryType() == cell.Tetrahedron_3d_1):
            cellTypes.append(tvtk.Tetra().cell_type)
        elif (_cell.giveGeometryType() == cell.Brick_3d_1):
            cellTypes.append(tvtk.Hexahedron().cell_type)
        else:
            raise TypeError("Unsupported cell type")
    # Create the array of cells unambiguously.
    cell_array = tvtk.CellArray()
    cell_array.set_cells(len(offsets), cells)

    # Now create the UG.
    ans = tvtk.UnstructuredGrid(points=points)
    # Now just set the cell types and reuse the ug locations and cells.
    ans.set_cells(cellTypes, offsets, cell_array)

    return ans

def attachPrimaryField(ug, _field):
    #t1=range(0,100,8)
    #temperature=array(t1)
    mesh=_field.giveMesh()
    if _field.giveValueType() == field.FieldValueType.vector:

        a=zeros((mesh.giveNumberOfVertices(),3))
        for vertex in mesh.vertices():
            (a[vertex.number,0],a[vertex.number,1],a[vertex.number,2]) = _field.evaluateAtVertex(vertex.number)

        ug.point_data.vectors=a
        ug.point_data.vectors.name = 'velocities'

    elif _field.giveValueType() == field.FieldValueType.tensor:
        a=zeros((mesh.giveNumberOfVertices(),9))
        for vertex in mesh.vertices():
            (a[vertex.number,0],a[vertex.number,1],a[vertex.number,2],
             a[vertex.number,3],a[vertex.number,4],a[vertex.number,5],
             a[vertex.number,6],a[vertex.number,7],a[vertex.number,8]) = _field.evaluateAtVertex(vertex.number)

        ug.point_data.tensors=a
        ug.point_data.tensors.name = 'tensors'
        
    elif _field.giveValueType() == field.FieldValueType.scalar:

        a=zeros((mesh.giveNumberOfVertices(),1))
        for vertex in mesh.vertices():
            a[vertex.number] = _field.evaluateAtVertex(vertex.number)

        ug.point_data.scalars=a
        ug.point_data.scalars.name = 'scalars'

    if debug: 
        print "vtk:", a
    
    return

def field2VTKDataSource (field):
    _ug = create_tvtk_ug(field.giveMesh())
    attachPrimaryField(_ug,field)
    return VTKDataSource(data = _ug)

def mesh2VTKDataSource (mesh):
    _ug = create_tvtk_ug(mesh)
    return VTKDataSource(data = _ug)
   
def mesh2VTKDatasetManager (mesh):
    _ug = create_tvtk_ug(mesh)
    return  DatasetManager(dataset=_ug)


""" 
Finds a real roots of quadratic equation:
ax^2 + bx + c = 0

By substituting x = y-t and t = a/2, 
the equation reduces to y^2 + (b-t^2) = 0 
which has easy solution y = +/- sqrt(t^2-b) 
""" 
def quadratic_real (a, b, c): 
    import math, cmath 
    if math.fabs(a) <= 1.e-10:
        if math.fabs(b) <= 1.e-10:
            return ()
        else:
            return (-c/float(b),)
    else:
        
        a, b = b / float(a), c / float(a) 
        t = a / 2.0 
        r = t**2 - b 
        if r >= 0: # real roots 
            y1 = math.sqrt(r) 
        else: # complex roots 
            return ()
        y2 = -y1 
        return (y1 - t, y2 - t)


import vertex
"""
Creates new unstructured mesh from given oofem input file. Does not uses oofem.api.
"""
def readOfemMesh(oofemInputFileName):
    def giveNextLine (file):
        """Returns next record (line) from oofem input file"""
        # just skip comments
        cond = False
        while (not cond):
            line=file.readline()
            cond=((line != "") and (line[0]!='#'))
            #print cond, line
        return line
        
    with open(oofemInputFileName, 'r') as f:
        # skip output file record+job dscription rec and read engngm record
        for i in xrange(3):
            rec=giveNextLine(f)
        nmodules = 0
        # rec is now engngm record; see if modules specified
        match = re.search('nmodules\s+(\d+)', rec, re.IGNORECASE)
        if match:
            nmodules = int(match.group(1))
        #skip all modules + domain rec+ output manager rec
        for i in xrange(nmodules+2):
            rec=giveNextLine(f)
        #read component record
        rec=giveNextLine(f)
        # parse number of nodes and number of elements
        match = re.search('ndofman\s+(\d+)', rec, re.IGNORECASE)
        if match:
            nnode = int(match.group(1))
        else:
            print "readOfemMesh: oofem input file consistency error: component record not found:\n", rec
            return None
        match = re.search('nelem\s+(\d+)', rec, re.IGNORECASE)
        if match:
            nelem = int(match.group(1))
        else:
            print "readOfemMesh: oofem input file consistency error: component record not found:\n", rec
            return None
        vertexList=[]
        cellList  =[]
        mymesh = mesh.UnstructuredMesh()
        # ok parse next nnode records
        for i in xrange(nnode):
            label = i
            rec = giveNextLine(f)
            # parse node record, get label and coordinates
            match = re.search('globnum\s+(\d+)\s+', rec, re.IGNORECASE)
            if match:
                label=int(match.group(1))
            # 2 coordinates provided
            match = re.search('coords\s+2\s+(?P<c1>[-]*\d+\.\d+(e[+-]\d+)?)\s+(?P<c2>[-]*\d+\.\d+(e[+-]\d+)?)', rec, re.IGNORECASE)
            if match:
                coords=(float(match.group('c1')), float(match.group('c2')), 0.0)
            else:
                match = re.search('coords\s+3\s+(?P<c1>[-]*\d+\.\d+(e[+-]\d+)?)\s+(?P<c2>[-]*\d+\.\d+(e[+-]\d+)?)\s+(?P<c3>[-]*\d+\.\d+(e[+-]\d+)?)', rec, re.IGNORECASE)
                if match:
                    coords=(float(match.group('c1')), float(match.group('c2')), float(match.group('c3')))
                else:
                    print "readOfemMesh: dofman consistency error: coords not found in ", rec
                    return None
            vertexList.append(vertex.Vertex(i, label, coords))
            
        #parse elements
        for i in xrange(nelem):
            label = i
            rec=giveNextLine(f)
            # parse element record, get cell name
            match = re.search('^(\w+)', rec, re.IGNORECASE)
            if match:
                cellName=match.group(1).lower()
            else:
                print "readOfemMesh: unable to match cell type in rec: ", rec
                return None
            
            #get label
            match = re.search('globnum\s+(\d+)\s+', rec, re.IGNORECASE)
            if match:
                label=int(match.group(1))
            #get list of nodes
            match = re.search('nodes\s+(\d+)\s+([\d\s]+)+', rec, re.IGNORECASE)
            if match:
                ncellNodes = int(match.group(1))
                cellNodes =tuple([int(i)-1 for i in re.findall(r'(\d+)\s+', match.group(2))])
            else:
                print "readOfemMesh: missing nodes keyword in element rec: ", rec
                return None
            #create new cell
            if (cellName == 'lspace'):
                c = cell.Brick_3d_lin(mymesh, i, label, cellNodes)
            elif (cellName == 'ltrspace'):
                c = cell.Tetrahedron_3d_lin(mymesh, i, label, cellNodes)
            elif (cellName == 'l4axisymm'):
                c = cell.Quad_2d_lin(mymesh, i, label, cellNodes)
            else:
                print "readOfemMesh: unrecognized element type ", cellName," in ", rec
                return None
            cellList.append(c)
        mymesh.setup(vertexList, cellList)
        #print mymesh

    return mymesh            
            
        
